<?php
/**
 * Plugin Name: Socialwire Article Generator
 * Plugin URI: https://gas.cowriter.jp/
 * Description: AI-powered WordPress plugin that automatically generates articles from press releases to support efficient media management.
 * Version: 1.1.12
 * Author: SocialWire
 * Author URI: https://www.socialwire.net/
 * License: GPL v2 or later
 * Text Domain: socialwire-article-generator
 * Domain Path: /languages
 * Requires PHP: 7.4
 * Requires at least: 5.0
 *
 * @package Socialwire_Article_Generator
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin class for Socialwire Article Generator.
 *
 * Handles RSS import, article generation, and WordPress integration
 * for the Socialwire Article Generator plugin.
 * Memory optimized version with conditional loading.
 *
 * @since 1.0.0
 */
class Socialwire_Article_Generator {

	/**
	 * Option name for plugin settings.
	 *
	 * @var string
	 */
	private $option_name = 'socialwire_generator_settings';

	/**
	 * Option name for unique ID storage.
	 *
	 * @var string
	 */
	private $unique_id_option = 'socialwire_generator_unique_id';

	/**
	 * Lock timeout in seconds (30 minutes).
	 *
	 * @var int
	 */
	private $lock_timeout = 1800;

	/**
	 * Debug logs array for manual execution.
	 *
	 * @var array Debug logs for manual execution.
	 */
	private $debug_logs = array();

	/**
	 * Static flag to prevent multiple text domain loads.
	 *
	 * @var bool
	 */
	private static $text_domain_loaded = false;

	/**
	 * Memory usage tracking checkpoints.
	 *
	 * @var array
	 */
	private $memory_checkpoints = array();

	/**
	 * Initial memory usage when plugin started.
	 *
	 * @var int
	 */
	private $initial_memory = 0;

	/**
	 * Get plugin version from plugin data.
	 *
	 * @return string Plugin version.
	 */
	private function get_plugin_version() {
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugin_data = get_plugin_data( __FILE__ );
		return $plugin_data['Version'];
	}

	/**
	 * Constructor. Initialize hooks and actions. Memory optimized version.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Start memory tracking.
		$this->initial_memory = memory_get_usage(true);
		$this->add_memory_checkpoint('plugin_start');

		// Conditional initialization to reduce memory usage.
		if ( function_exists( 'is_admin' ) && is_admin() ) {
			$this->add_memory_checkpoint('before_admin_init');
			$this->init_admin();
			$this->add_memory_checkpoint('after_admin_init');
		} else {
			$this->add_memory_checkpoint('before_frontend_init');
			$this->init_frontend();
			$this->add_memory_checkpoint('after_frontend_init');
		}

		// Register scheduled actions.
		add_action( 'socialwire_scheduled_generate', array( $this, 'scheduled_generate' ) );
		add_action( 'socialwire_scheduled_rss_fetch', array( $this, 'scheduled_rss_fetch' ) );
		add_action( 'socialwire_scheduled_url_notify', array( $this, 'scheduled_url_notify' ) );
		add_filter( 'cron_schedules', array( $this, 'add_cron_interval' ) );

		// External cron endpoint (query parameter method).
		add_action( 'init', array( $this, 'handle_external_cron' ) );

		// Plugin lifecycle hooks.
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
		register_uninstall_hook( __FILE__, array( $this, 'uninstall' ) );
		
		$this->add_memory_checkpoint('constructor_end');
	}

	/**
	 * Add memory checkpoint for tracking.
	 *
	 * @param string $label Checkpoint label.
	 * @since 1.1.2
	 */
	private function add_memory_checkpoint($label) {
		$this->memory_checkpoints[$label] = array(
			'memory_usage' => memory_get_usage(true),
			'peak_memory' => memory_get_peak_usage(true),
			'time' => microtime(true)
		);
	}

	/**
	 * Get current memory usage information.
	 *
	 * @return array Memory usage information.
	 * @since 1.1.2
	 */
	public function get_memory_usage_info() {
		$current_memory = memory_get_usage(true);
		$peak_memory = memory_get_peak_usage(true);
		$plugin_memory_usage = $current_memory - $this->initial_memory;
		
		return array(
			'current' => $current_memory,
			'peak' => $peak_memory,
			'plugin_usage' => $plugin_memory_usage,
			'initial' => $this->initial_memory,
			'checkpoints_count' => count($this->memory_checkpoints)
		);
	}

	/**
	 * Get formatted memory usage report for display.
	 *
	 * @return string Formatted memory usage report.
	 * @since 1.1.2
	 */
	public function get_memory_usage_display() {
		$memory_info = $this->get_memory_usage_info();
		
		$output = '<div style="padding: 2px;">';
		$output .= '<strong>' . esc_html__('Plugin Memory Usage:', 'socialwire-article-generator') . '</strong><br>';
		$output .= sprintf(
			'<span style="color: #0073aa;">%s:</span> %s<br>',
			esc_html__('Plugin Memory Used', 'socialwire-article-generator'),
			size_format($memory_info['plugin_usage'])
		);
		$output .= sprintf(
			'<span style="color: #135e96;">%s:</span> %s<br>',
			esc_html__('Current Total Memory', 'socialwire-article-generator'),
			size_format($memory_info['current'])
		);
		$output .= sprintf(
			'<span style="color: #d54e21;">%s:</span> %s<br>',
			esc_html__('Peak Memory Usage', 'socialwire-article-generator'),
			size_format($memory_info['peak'])
		);
		$output .= sprintf(
			'<span style="color: #81b300;">%s:</span> %d',
			esc_html__('Memory Checkpoints', 'socialwire-article-generator'),
			$memory_info['checkpoints_count']
		);
		$output .= '</div>';
		
		return $output;
	}

	/**
	 * Initialize admin-specific functionality.
	 *
	 * @since 1.1.2
	 */
	private function init_admin() {
		// Load text domain early on plugins_loaded hook for better translation support.
		add_action( 'plugins_loaded', array( $this, 'load_text_domain' ), 1 );
		
		// Additional safety: Load on admin_init as well.
		add_action( 'admin_init', array( $this, 'load_text_domain' ), 1 );
		
		// Only load heavy resources when actually on plugin settings page.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is for admin page detection only
		if ( isset( $_GET['page'] ) && $_GET['page'] === 'socialwire-generator-settings' ) {
			if ( function_exists( 'wp_raise_memory_limit' ) ) {
				wp_raise_memory_limit( 'admin' );
			}
		}

		add_action( 'admin_menu', array( $this, 'admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		add_action( 'wp_ajax_socialwire_manual_generate', array( $this, 'ajax_manual_generate' ) );
		add_action( 'wp_ajax_socialwire_manual_fetch', array( $this, 'ajax_manual_fetch' ) );

		// Manual change detection hooks only in admin.
		add_action( 'post_updated', array( $this, 'detect_manual_edit' ), 10, 3 );
		add_action( 'wp_trash_post', array( $this, 'detect_manual_delete' ), 10, 1 );
		add_action( 'before_delete_post', array( $this, 'detect_manual_delete' ), 10, 2 );
	}

	/**
	 * Initialize frontend-specific functionality.
	 *
	 * @since 1.1.2
	 */
	private function init_frontend() {
		// Only load necessary frontend features.
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ) );
		add_action( 'wp_footer', array( $this, 'add_modal_html' ) );
		add_filter( 'the_content', array( $this, 'filter_content_links' ), 10, 1 );
	}

	/**
	 * Handle external cron requests via query parameter.
	 *
	 * URL format: https://example.com/?socialwire_cron=1&token=xxx&action=all
	 * Actions: generate, fetch, notify, all (default)
	 *
	 * @since 1.1.12
	 */
	public function handle_external_cron() {
		// Check query parameter.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- External cron uses token authentication.
		if ( ! isset( $_GET['socialwire_cron'] ) || $_GET['socialwire_cron'] !== '1' ) {
			return;
		}

		// Get and validate security token.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- External cron uses token authentication.
		$token = isset( $_GET['token'] ) ? sanitize_text_field( wp_unslash( $_GET['token'] ) ) : '';
		$stored_token = get_option( 'socialwire_cron_token' );

		// Generate token if not set.
		if ( empty( $stored_token ) ) {
			$stored_token = wp_generate_password( 32, false );
			update_option( 'socialwire_cron_token', $stored_token );
		}

		// Validate token.
		if ( ! hash_equals( $stored_token, $token ) ) {
			status_header( 403 );
			echo 'Forbidden: Invalid token';
			exit;
		}

		// Get action to execute.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- External cron uses token authentication.
		$action = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : 'all';

		// Check force parameter (skip auto_generate and time checks).
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- External cron uses token authentication.
		$force = isset( $_GET['force'] ) && $_GET['force'] === '1';

		// Output header.
		header( 'Content-Type: text/plain; charset=utf-8' );

		$results = array();

		switch ( $action ) {
			case 'generate':
				$result = $this->execute_external_cron_generate( $force );
				$results[] = $result;
				break;

			case 'fetch':
				$result = $this->execute_external_cron_fetch( $force );
				$results[] = $result;
				break;

			case 'notify':
				$result = $this->execute_external_cron_notify( $force );
				$results[] = $result;
				break;

			case 'all':
			default:
				$result = $this->execute_external_cron_generate( $force );
				$results[] = $result;

				$result = $this->execute_external_cron_fetch( $force );
				$results[] = $result;

				$result = $this->execute_external_cron_notify( $force );
				$results[] = $result;
				break;
		}

		echo esc_html( implode( "\n", $results ) ) . "\n";
		echo 'Completed at: ' . esc_html( current_time( 'Y-m-d H:i:s' ) );
		exit;
	}

	/**
	 * Get external cron URL for display in settings.
	 *
	 * @return string External cron URL with token.
	 * @since 1.1.12
	 */
	public function get_external_cron_url() {
		$token = get_option( 'socialwire_cron_token' );

		// Generate token if not set.
		if ( empty( $token ) ) {
			$token = wp_generate_password( 32, false );
			update_option( 'socialwire_cron_token', $token );
		}

		return add_query_arg(
			array(
				'socialwire_cron' => '1',
				'token'           => $token,
			),
			home_url( '/' )
		);
	}

	/**
	 * Regenerate external cron token.
	 *
	 * @return string New token.
	 * @since 1.1.12
	 */
	public function regenerate_cron_token() {
		$new_token = wp_generate_password( 32, false );
		update_option( 'socialwire_cron_token', $new_token );
		return $new_token;
	}

	/**
	 * Execute article generation for external cron.
	 *
	 * @param bool $force Whether to force execution regardless of settings.
	 * @return string Result message.
	 * @since 1.1.12
	 */
	private function execute_external_cron_generate( $force = false ) {
		$this->socialwire_log( '=== EXTERNAL CRON GENERATE STARTED (force=' . ( $force ? 'true' : 'false' ) . ') ===' );

		$lock_id = $this->acquire_distributed_lock( 'article_generate', 900 );
		if ( ! $lock_id ) {
			$this->socialwire_log( 'Another instance is running article generation, skipping' );
			return '[SKIP] Article generation: Another instance is running';
		}

		try {
			$settings = get_option( $this->option_name, array() );
			$auto_generate = isset( $settings['auto_generate'] ) && $settings['auto_generate'];
			$auto_exec_mode = isset( $settings['auto_exec_mode'] ) ? $settings['auto_exec_mode'] : 'interval';

			// ウェブフックモードの場合は自動的にforce扱い（設定で明示的にウェブフックを選んでいるため）
			if ( 'webhook' === $auto_exec_mode && $auto_generate ) {
				$force = true;
			}

			if ( ! $force && ! $auto_generate ) {
				$this->socialwire_log( 'Auto generation is disabled and force=false' );
				return '[SKIP] Article generation: Auto generation is disabled (add &force=1 to override)';
			}

			if ( ! $force ) {
				// Check time restriction only when not forced.
				if ( 'hourly' === $auto_exec_mode ) {
					$auto_exec_hours = isset( $settings['auto_exec_hours'] ) ? $settings['auto_exec_hours'] : array();
					$current_hour = intval( current_time( 'G' ) );
					
					if ( ! in_array( $current_hour, $auto_exec_hours, true ) ) {
						$this->socialwire_log( sprintf( 'Current hour (%d) is not in execution schedule', $current_hour ) );
						return sprintf( '[SKIP] Article generation: Current hour (%d) is not in schedule (add &force=1 to override)', $current_hour );
					}
				}
			}

			$result = $this->execute_generate_process();
			$this->socialwire_log( 'Generate process result: ' . $result['message'] );
			return '[OK] Article generation: ' . $result['message'];

		} finally {
			$this->release_distributed_lock( 'article_generate', $lock_id );
			$this->socialwire_log( '=== EXTERNAL CRON GENERATE FINISHED ===' );
		}
	}

	/**
	 * Execute RSS fetch for external cron.
	 *
	 * @param bool $force Whether to force execution regardless of settings.
	 * @return string Result message.
	 * @since 1.1.12
	 */
	private function execute_external_cron_fetch( $force = false ) {
		$this->socialwire_log( '=== EXTERNAL CRON RSS FETCH STARTED (force=' . ( $force ? 'true' : 'false' ) . ') ===' );

		$lock_id = $this->acquire_distributed_lock( 'rss_fetch', 900 );
		if ( ! $lock_id ) {
			$this->socialwire_log( 'Another instance is running RSS fetch, skipping' );
			return '[SKIP] RSS fetch: Another instance is running';
		}

		try {
			$settings = get_option( $this->option_name, array() );
			$auto_generate = isset( $settings['auto_generate'] ) && $settings['auto_generate'];
			$auto_exec_mode = isset( $settings['auto_exec_mode'] ) ? $settings['auto_exec_mode'] : 'interval';

			// ウェブフックモードの場合は自動的にforce扱い（設定で明示的にウェブフックを選んでいるため）
			if ( 'webhook' === $auto_exec_mode && $auto_generate ) {
				$force = true;
			}

			if ( ! $force && ! $auto_generate ) {
				$this->socialwire_log( 'Auto generation is disabled and force=false' );
				return '[SKIP] RSS fetch: Auto generation is disabled (add &force=1 to override)';
			}

			if ( ! $force ) {
				// Check time restriction only when not forced.
				if ( 'hourly' === $auto_exec_mode ) {
					$auto_exec_hours = isset( $settings['auto_exec_hours'] ) ? $settings['auto_exec_hours'] : array();
					$current_hour = intval( current_time( 'G' ) );
					
					if ( ! in_array( $current_hour, $auto_exec_hours, true ) ) {
						$this->socialwire_log( sprintf( 'Current hour (%d) is not in execution schedule', $current_hour ) );
						return sprintf( '[SKIP] RSS fetch: Current hour (%d) is not in schedule (add &force=1 to override)', $current_hour );
					}
				}
			}

			$result = $this->execute_rss_fetch_process();
			$this->socialwire_log( 'RSS fetch result: ' . $result['message'] );
			return '[OK] RSS fetch: ' . $result['message'];

		} finally {
			$this->release_distributed_lock( 'rss_fetch', $lock_id );
			$this->socialwire_log( '=== EXTERNAL CRON RSS FETCH FINISHED ===' );
		}
	}

	/**
	 * Execute URL notification for external cron.
	 *
	 * @param bool $force Whether to force execution regardless of settings.
	 * @return string Result message.
	 * @since 1.1.12
	 */
	private function execute_external_cron_notify( $force = false ) {
		$this->socialwire_log( '=== EXTERNAL CRON URL NOTIFY STARTED (force=' . ( $force ? 'true' : 'false' ) . ') ===' );

		$lock_id = $this->acquire_distributed_lock( 'url_notify', 900 );
		if ( ! $lock_id ) {
			$this->socialwire_log( 'Another instance is running URL notify, skipping' );
			return '[SKIP] URL notify: Another instance is running';
		}

		try {
			$settings = get_option( $this->option_name, array() );
			$auto_generate = isset( $settings['auto_generate'] ) && $settings['auto_generate'];
			$auto_exec_mode = isset( $settings['auto_exec_mode'] ) ? $settings['auto_exec_mode'] : 'interval';

			// ウェブフックモードの場合は自動的にforce扱い（設定で明示的にウェブフックを選んでいるため）
			if ( 'webhook' === $auto_exec_mode && $auto_generate ) {
				$force = true;
			}

			if ( ! $force && ! $auto_generate ) {
				$this->socialwire_log( 'Auto generation is disabled and force=false' );
				return '[SKIP] URL notify: Auto generation is disabled (add &force=1 to override)';
			}

			if ( ! $force ) {
				// Check time restriction only when not forced.
				if ( 'hourly' === $auto_exec_mode ) {
					$auto_exec_hours = isset( $settings['auto_exec_hours'] ) ? $settings['auto_exec_hours'] : array();
					$current_hour = intval( current_time( 'G' ) );
					
					if ( ! in_array( $current_hour, $auto_exec_hours, true ) ) {
						$this->socialwire_log( sprintf( 'Current hour (%d) is not in execution schedule', $current_hour ) );
						return sprintf( '[SKIP] URL notify: Current hour (%d) is not in schedule (add &force=1 to override)', $current_hour );
					}
				}
			}

			$result = $this->execute_url_notify_process();
			$this->socialwire_log( 'URL notify result: ' . $result['message'] );
			return '[OK] URL notify: ' . $result['message'];

		} finally {
			$this->release_distributed_lock( 'url_notify', $lock_id );
			$this->socialwire_log( '=== EXTERNAL CRON URL NOTIFY FINISHED ===' );
		}
	}

	/**
	 * Override admin title for proper translation. Memory optimized.
	 *
	 * @param string $admin_title The admin title.
	 * @param string $title The page title.
	 * @return string Modified admin title.
	 */
	public function override_admin_title( $admin_title, $title ) {
		$current_screen = get_current_screen();
		
		if ( $current_screen && $current_screen->id === 'settings_page_socialwire-generator-settings' ) {
			// Load text domain only when needed.
			$this->load_text_domain();
			$translated_title = __( 'Socialwire Article Generator Settings', 'socialwire-article-generator' );
			$admin_title = str_replace( 'Socialwire Article Generator Settings', $translated_title, $admin_title );
		}
		
		return $admin_title;
	}

	/**
	 * Force user locale for MO file loading. Simplified version.
	 *
	 * @param string $mofile The path to the MO file.
	 * @param string $domain The text domain.
	 * @return string Modified MO file path.
	 */
	public function force_user_locale_mofile( $mofile, $domain ) {
		// Only apply to our plugin's text domain.
		if ( 'socialwire-article-generator' !== $domain ) {
			return $mofile;
		}
		
		// Only process in admin with user locale support.
		if ( is_admin() && function_exists( 'get_user_locale' ) ) {
			$user_locale = get_user_locale();
			$site_locale = get_locale();
			
			if ( $user_locale !== $site_locale ) {
				$plugin_rel_path = dirname( plugin_basename( __FILE__ ) ) . '/languages';
				$user_mofile = WP_PLUGIN_DIR . '/' . $plugin_rel_path . '/' . $user_locale . '.mo';
				
				if ( file_exists( $user_mofile ) ) {
					return $user_mofile;
				}
			}
		}
		
		return $mofile;
	}

	/**
	 * Force user locale for our plugin. Simplified version.
	 *
	 * @param string $locale The current locale.
	 * @return string Modified locale.
	 */
	public function force_user_locale( $locale ) {
		// Only apply on plugin settings page.
		if ( is_admin() && function_exists( 'get_user_locale' ) ) {
			$current_screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
			if ( $current_screen && $current_screen->id === 'settings_page_socialwire-generator-settings' ) {
				return get_user_locale();
			}
		}
		
		return $locale;
	}

	/**
	 * Conditionally load text domain only when on plugin page to save memory.
	 *
	 * @since 1.1.2
	 */
	public function conditional_load_text_domain() {
		// Only load translations when actually on plugin settings page.
		$screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
		if ( $screen && $screen->id === 'settings_page_socialwire-generator-settings' ) {
			$this->load_text_domain();
		}
	}

	/**
	 * Load text domain for translations. Fixed version for Japanese support.
	 *
	 * @since 1.1.2
	 * @return bool True if text domain was loaded successfully.
	 */
	public function load_text_domain() {
		$domain = 'socialwire-article-generator';
		
		// Force reload by unloading existing domain first
		if ( is_textdomain_loaded( $domain ) ) {
			unload_textdomain( $domain );
		}

		// Priority 1: Get user locale first (more important than site locale)
		$locale = 'en_US'; // Default fallback
		if ( is_admin() && function_exists( 'get_user_locale' ) ) {
			$user_locale = get_user_locale();
			if ( $user_locale ) {
				$locale = $user_locale;
			}
		}
		
		// Fallback to site locale if user locale not available
		if ( $locale === 'en_US' && function_exists( 'determine_locale' ) ) {
			$locale = determine_locale();
		}
		
		// Define plugin directory paths
		$plugin_dir = dirname( __FILE__ );
		
		// Initialize result variable
		$result = false;
		
		// Method 1: Force Japanese loading if user wants Japanese
		if ( $locale === 'ja' || strpos( $locale, 'ja' ) === 0 ) {
			$ja_mo_file = $plugin_dir . '/languages/ja.mo';
			if ( file_exists( $ja_mo_file ) ) {
				$result = load_textdomain( $domain, $ja_mo_file );
				if ( $result ) {
					$this->text_domain_loaded = true;
					return true;
				}
			}
		}
		
		// Method 2: Try direct loading with current locale
		if ( ! $result ) {
			$mo_file = $plugin_dir . '/languages/' . $locale . '.mo';
			if ( file_exists( $mo_file ) ) {
				$result = load_textdomain( $domain, $mo_file );
			}
		}
		
		// Update loading state
		$this->text_domain_loaded = $result;
		
		return $result;
	}

	/**
	 * Acquire distributed lock.
	 *
	 * @param string   $lock_name Lock name identifier.
	 * @param int|null $timeout   Lock timeout in seconds. Default null uses class timeout.
	 * @return string|false Lock ID on success, false on failure.
	 */
	private function acquire_distributed_lock( $lock_name, $timeout = null ) {
		if ( null === $timeout ) {
			$timeout = $this->lock_timeout;
		}

		$lock_key     = 'socialwire_lock_' . $lock_name;
		$current_time = time();

		delete_transient( $lock_key );

		$lock_value = array(
			'hostname'  => gethostname(),
			'pid'       => getmypid(),
			'time'      => $current_time,
			'unique_id' => uniqid( 'socialwire_', true ),
		);

		$success = set_transient( $lock_key, wp_json_encode( $lock_value ), $timeout );

		if ( $success ) {
			$this->socialwire_log( 'Successfully acquired lock: ' . $lock_name . ' (ID: ' . $lock_value['unique_id'] . ')' );
			return $lock_value['unique_id'];
		}

		$existing_lock = get_transient( $lock_key );
		if ( $existing_lock ) {
			$existing_data = json_decode( $existing_lock, true );
			$lock_age      = $current_time - ( $existing_data['time'] ?? 0 );
			$this->socialwire_log(
				sprintf(
					'Lock already exists: %s by %s (PID: %s, Age: %d seconds)',
					$lock_name,
					$existing_data['hostname'] ?? 'unknown',
					$existing_data['pid'] ?? 'unknown',
					$lock_age
				)
			);
		}

		return false;
	}

	/**
	 * Release distributed lock.
	 *
	 * @param string $lock_name Lock name identifier.
	 * @param string $lock_id   Lock ID to verify ownership.
	 * @return bool True if lock was released, false otherwise.
	 */
	private function release_distributed_lock( $lock_name, $lock_id ) {
		$lock_key = 'socialwire_lock_' . $lock_name;

		$existing_lock = get_transient( $lock_key );
		if ( $existing_lock ) {
			$existing_data = json_decode( $existing_lock, true );

			if ( isset( $existing_data['unique_id'] ) && $existing_data['unique_id'] === $lock_id ) {
				delete_transient( $lock_key );
				$this->socialwire_log( 'Released lock: ' . $lock_name . ' (ID: ' . $lock_id . ')' );
				return true;
			} else {
				$this->socialwire_log( 'Cannot release lock: not owned by this process' );
				return false;
			}
		}

		return true;
	}

	/**
	 * Get or generate a unique environment ID.
	 */
	private function get_environment_unique_id() {
		global $wpdb;

		$unique_id = get_option( $this->unique_id_option );

		if ( ! $unique_id ) {
			$lock_id = $this->acquire_distributed_lock( 'unique_id_generation', 60 );

			if ( $lock_id ) {
				try {
					$unique_id = get_option( $this->unique_id_option );

					if ( ! $unique_id ) {
						$components = array(
							get_site_url(),
							get_option( 'siteurl' ),
							DB_HOST,
							DB_NAME,
							'socialwire_generator_seed',
						);

						$new_id = hash( 'sha256', implode( '|', $components ) );

						// Use WordPress options API instead of direct database access.
						if ( ! get_option( $this->unique_id_option ) ) {
							add_option( $this->unique_id_option, $new_id, '', 'yes' );
						}

						$unique_id = get_option( $this->unique_id_option );

						$this->socialwire_log( 'Generated new shared PCN ID: ' . $unique_id );
					} else {
						$this->socialwire_log( 'PCN ID already exists (found after lock): ' . $unique_id );
					}
				} finally {
					$this->release_distributed_lock( 'unique_id_generation', $lock_id );
				}
			} else {
				sleep( 1 );
				$unique_id = get_option( $this->unique_id_option );

				if ( ! $unique_id ) {
					$this->socialwire_log( 'Warning: Could not acquire lock for PCN ID generation' );
					return false;
				}
			}
		}

		return $unique_id;
	}

	/**
	 * Add custom cron intervals.
	 *
	 * @param array $schedules Existing cron schedules.
	 * @return array Modified schedules array.
	 */
	public function add_cron_interval( $schedules ) {
		$schedules['socialwire_30min'] = array(
			'interval' => 1800,
			'display'  => __( 'Every 30 minutes', 'socialwire-article-generator' ),
		);
		$schedules['socialwire_1hour'] = array(
			'interval' => 3600,
			'display'  => __( 'Every 1 hour', 'socialwire-article-generator' ),
		);
		$schedules['socialwire_2hour'] = array(
			'interval' => 7200,
			'display'  => __( 'Every 2 hours', 'socialwire-article-generator' ),
		);
		$schedules['socialwire_3hour'] = array(
			'interval' => 10800,
			'display'  => __( 'Every 3 hours', 'socialwire-article-generator' ),
		);
		$schedules['socialwire_4hour'] = array(
			'interval' => 14400,
			'display'  => __( 'Every 4 hours', 'socialwire-article-generator' ),
		);
		$schedules['socialwire_5hour'] = array(
			'interval' => 18000,
			'display'  => __( 'Every 5 hours', 'socialwire-article-generator' ),
		);
		$schedules['socialwire_6hour'] = array(
			'interval' => 21600,
			'display'  => __( 'Every 6 hours', 'socialwire-article-generator' ),
		);
		$schedules['socialwire_12hour'] = array(
			'interval' => 43200,
			'display'  => __( 'Every 12 hours', 'socialwire-article-generator' ),
		);
		return $schedules;
	}

	/**
	 * Plugin activation hook. Set up scheduled events.
	 *
	 * @since 1.0.0
	 */
	public function activate() {
		$this->reschedule_cron_events();
		$this->get_environment_unique_id();

		// Send plugin user URL on activation
		$this->send_plugin_user_url('activation');

		add_filter( 'cron_schedules', array( $this, 'add_cron_interval' ) );
	}

	/**
	 * Reschedule cron events based on current settings.
	 *
	 * @since 1.1.8
	 */
	private function reschedule_cron_events() {
		// Clear existing schedules
		wp_clear_scheduled_hook( 'socialwire_scheduled_generate' );
		wp_clear_scheduled_hook( 'socialwire_scheduled_rss_fetch' );
		wp_clear_scheduled_hook( 'socialwire_scheduled_url_notify' );

		$settings = get_option( $this->option_name, array() );
		$auto_exec_mode = isset( $settings['auto_exec_mode'] ) ? $settings['auto_exec_mode'] : 'interval';

		// ウェブフックモードの場合はcronをスケジュールしない（外部cronで制御）
		if ( 'webhook' === $auto_exec_mode ) {
			$this->socialwire_log( 'Webhook mode: Skipping cron scheduling (controlled by external cron)' );
			return;
		}

		if ( 'interval' === $auto_exec_mode ) {
			// 繰り返し実行モード
			$interval = isset( $settings['auto_exec_interval'] ) ? $settings['auto_exec_interval'] : '30min';
			$interval_name = 'socialwire_' . $interval;
			$next_run = time() + 60;

			if ( ! wp_next_scheduled( 'socialwire_scheduled_generate' ) ) {
				wp_schedule_event( $next_run, $interval_name, 'socialwire_scheduled_generate' );
			}

			if ( ! wp_next_scheduled( 'socialwire_scheduled_rss_fetch' ) ) {
				wp_schedule_event( $next_run + 300, $interval_name, 'socialwire_scheduled_rss_fetch' );
			}

			if ( ! wp_next_scheduled( 'socialwire_scheduled_url_notify' ) ) {
				wp_schedule_event( $next_run + 600, $interval_name, 'socialwire_scheduled_url_notify' );
			}
		} else if ( 'hourly' === $auto_exec_mode ) {
			// 時間指定モード
			$hours = isset( $settings['auto_exec_hours'] ) ? $settings['auto_exec_hours'] : array();
			if ( ! empty( $hours ) ) {
				$this->schedule_hourly_events( $hours );
			}
		}
	}

	/**
	 * Schedule events for specific hours.
	 *
	 * @param array $hours Array of hours (0-23) to run events.
	 * @since 1.1.8
	 */
	private function schedule_hourly_events( $hours ) {
		if ( empty( $hours ) ) {
			return;
		}

		// 時間指定モードでは hourly で設定し、scheduled_generate内で時刻チェック
		$next_run = time() + 60;

		if ( ! wp_next_scheduled( 'socialwire_scheduled_generate' ) ) {
			wp_schedule_event( $next_run, 'hourly', 'socialwire_scheduled_generate' );
		}

		if ( ! wp_next_scheduled( 'socialwire_scheduled_rss_fetch' ) ) {
			wp_schedule_event( $next_run + 300, 'hourly', 'socialwire_scheduled_rss_fetch' );
		}

		if ( ! wp_next_scheduled( 'socialwire_scheduled_url_notify' ) ) {
			wp_schedule_event( $next_run + 600, 'hourly', 'socialwire_scheduled_url_notify' );
		}
	}

	/**
	 * Plugin deactivation hook. Clear scheduled events.
	 *
	 * @since 1.0.0
	 */
	public function deactivate() {
		wp_clear_scheduled_hook( 'socialwire_scheduled_generate' );
		wp_clear_scheduled_hook( 'socialwire_scheduled_rss_fetch' );
		wp_clear_scheduled_hook( 'socialwire_scheduled_url_notify' );
	}

	/**
	 * Plugin uninstall hook. Clean up options and transients.
	 *
	 * @since 1.0.0
	 */
	public static function uninstall() {
		delete_option( 'pcn_generator_settings' );
		delete_option( 'pcn_generator_unique_id' );
		delete_option( 'pcn_rss_last_modified' );
		delete_option( 'pcn_rss_latest_pub_date' );
		delete_option( 'socialwire_featured_image_settings' );
		delete_option( 'socialwire_content_image_settings' );
		delete_option( 'pcn_text_settings' );
		delete_option( 'pcn_manually_deleted_document_ids' );

		global $wpdb;
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_pcn_lock_%'" );
		$wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_pcn_lock_%'" );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
	}

	/**
	 * Add admin menu item for plugin settings. Fixed for Japanese translation.
	 *
	 * @since 1.0.0
	 */
	public function admin_menu() {
		// Ensure text domain is loaded before menu creation.
		$this->load_text_domain();
		
		$page_title = __( 'Socialwire Article Generator Settings', 'socialwire-article-generator' );
		$menu_title = __( 'Socialwire Article Generator', 'socialwire-article-generator' );
		
		add_options_page(
			$page_title,
			$menu_title,
			'manage_options',
			'socialwire-generator-settings',
			array( $this, 'settings_page' )
		);
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @param string $hook Current admin page hook.
	 * @since 1.0.0
	 */
	public function admin_enqueue_scripts( $hook ) {
		if ( 'settings_page_socialwire-generator-settings' !== $hook ) {
			return;
		}

		wp_enqueue_script( 'jquery' );

		// Register and enqueue admin CSS.
		$admin_css_handle = 'socialwire-admin-css';
		wp_register_style( $admin_css_handle, false );
		wp_enqueue_style( $admin_css_handle );
		wp_add_inline_style( $admin_css_handle, $this->get_admin_css() );

		// CSS-only implementation - no JavaScript needed for UI toggles
	}

	/**
	 * Extract lead text from content (before the first heading).
	 *
	 * @param string $content Content to process.
	 * @return array Array with 'lead_text' and 'main_content'.
	 */
	private function extract_lead_text( $content ) {
		if ( empty( $content ) ) {
			return array(
				'lead_text'    => '',
				'main_content' => $content,
			);
		}

		$this->socialwire_log( 'Extracting lead text from content' );

		// Find the first heading (h1-h6)
		$pattern = '/<h[1-6][^>]*>/i';
		$matches = array();
		
		if ( preg_match( $pattern, $content, $matches, PREG_OFFSET_CAPTURE ) ) {
			// Found a heading
			$heading_position = $matches[0][1];
			
			// Split content at the first heading
			$lead_text = substr( $content, 0, $heading_position );
			$main_content = substr( $content, $heading_position );
			
			// Clean up lead text (remove surrounding paragraphs and whitespace)
			$lead_text = trim( $lead_text );
			
			// Remove wrapping <p> tags if present
			if ( preg_match( '/^<p[^>]*>(.*)<\/p>$/s', $lead_text, $p_matches ) ) {
				$lead_text = trim( $p_matches[1] );
			}
			
			// Remove multiple <p> tags and convert to plain text with line breaks
			$lead_text = preg_replace( '/<\/p>\s*<p[^>]*>/', "\n\n", $lead_text );
			$lead_text = preg_replace( '/<p[^>]*>/', '', $lead_text );
			$lead_text = preg_replace( '/<\/p>/', '', $lead_text );
			$lead_text = wp_strip_all_tags( $lead_text );
			$lead_text = trim( $lead_text );
			
			$this->socialwire_log( 'Lead text extracted: ' . substr( $lead_text, 0, 100 ) . '...' );
			$this->socialwire_log( 'Main content starts with heading at position: ' . $heading_position );
			
			return array(
				'lead_text'    => $lead_text,
				'main_content' => $main_content,
			);
		}
		
		// No heading found, entire content is considered main content
		$this->socialwire_log( 'No heading found in content, no lead text to extract' );
		
		return array(
			'lead_text'    => '',
			'main_content' => $content,
		);
	}

	/**
	 * Get available custom fields grouped by type.
	 *
	 * @return array Array of custom fields grouped by type.
	 */
	private function get_available_custom_fields() {
		global $wpdb;

		// Get all custom field keys from the database
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
		$custom_field_keys = $wpdb->get_col(
			"SELECT DISTINCT meta_key FROM {$wpdb->postmeta} 
			WHERE meta_key NOT LIKE '\_%' 
			AND meta_key NOT LIKE 'pcn_%' 
			AND meta_key != 'field_%'
			ORDER BY meta_key"
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching

		$image_fields = array();
		$text_fields = array();

		// If ACF is active, get ACF field information
		if ( function_exists( 'get_field_objects' ) && function_exists( 'acf_get_fields' ) ) {
			$this->socialwire_log( 'ACF detected, getting field information' );
			
			// Get fields from ACF
			$field_groups = acf_get_field_groups();
			foreach ( $field_groups as $field_group ) {
				$fields = acf_get_fields( $field_group['key'] );
				if ( $fields ) {
					foreach ( $fields as $field ) {
						$field_name = $field['name'];
						$field_type = $field['type'];
						$field_label = isset( $field['label'] ) ? $field['label'] : $field_name;

						// Categorize by field type
						if ( in_array( $field_type, array( 'image', 'file' ) ) ) {
							$image_fields[ $field_name ] = sprintf( '%s (%s)', $field_label, $field_type );
						} elseif ( in_array( $field_type, array( 'text', 'textarea', 'wysiwyg', 'email', 'url', 'number' ) ) ) {
							$text_fields[ $field_name ] = sprintf( '%s (%s)', $field_label, $field_type );
						}
					}
				}
			}
		}

		// Add custom fields found in database that aren't in ACF
		foreach ( $custom_field_keys as $field_key ) {
			if ( ! isset( $image_fields[ $field_key ] ) && ! isset( $text_fields[ $field_key ] ) ) {
				// For non-ACF fields, we can't determine type, so add to both categories with a note
				$text_fields[ $field_key ] = $field_key . ' (' . __( 'Custom Field', 'socialwire-article-generator' ) . ')';
			}
		}

		return array(
			'image' => $image_fields,
			'text'  => $text_fields,
		);
	}

	/**
	 * Build category data with hierarchy information.
	 *
	 * @param array $selected_category_ids Array of category IDs.
	 * @return array Array of category data with hierarchy information.
	 */
	private function build_category_data( $selected_category_ids ) {
		if ( empty( $selected_category_ids ) ) {
			return array();
		}

		$categories = array();
		foreach ( $selected_category_ids as $cat_id ) {
			$category = get_category( $cat_id );
			if ( ! $category ) {
				continue;
			}

			$hierarchy_parts = array();
			$current_cat     = $category;

			while ( $current_cat ) {
				array_unshift( $hierarchy_parts, $current_cat->name );
				if ( $current_cat->parent ) {
					$current_cat = get_category( $current_cat->parent );
				} else {
					break;
				}
			}

			$categories[] = array(
				'id'          => $category->term_id,
				'name'        => $category->name,
				'parent_name' => $category->parent ? get_category( $category->parent )->name : null,
				'hierarchy'   => implode( ' > ', $hierarchy_parts ),
			);
		}

		return $categories;
	}

	/**
	 * Get admin CSS styles.
	 *
	 * @return string CSS code for admin interface.
	 * @since 1.1.0
	 */
	private function get_admin_css() {
		return '
			/* Accordion styling using details/summary. */
			.socialwire-accordion {
				margin-top: 20px;
				border: 1px solid #ccc;
				border-radius: 4px;
			}
			
			.socialwire-accordion-header {
				cursor: pointer;
				padding: 10px 15px;
				background-color: #f1f1f1;
				border: none;
				border-radius: 4px 4px 0 0;
				font-size: 14px;
				font-weight: 600;
				margin: 0;
				list-style: none;
				transition: background-color 0.2s ease;
				position: relative;
			}
			
			.socialwire-accordion-header:hover {
				background-color: #e8e8e8;
			}
			
			/* Custom arrow using CSS. */
			.socialwire-accordion-header::before {
				content: "▶";
				position: absolute;
				right: 15px;
				top: 50%;
				transform: translateY(-50%);
				transition: transform 0.2s ease;
			}
			
			.socialwire-accordion[open] .socialwire-accordion-header::before {
				transform: translateY(-50%) rotate(90deg);
			}
			
			/* Hide default disclosure triangle. */
			.socialwire-accordion-header::-webkit-details-marker {
				display: none;
			}
			
			.socialwire-accordion-content {
				padding: 20px;
				background-color: #fafafa;
				border-top: 1px solid #ddd;
				border-radius: 0 0 4px 4px;
			}
			
			/* Smooth animation for opening/closing. */
			.socialwire-accordion[open] .socialwire-accordion-content {
				animation: slideDown 0.2s ease-out;
			}
			
			@keyframes slideDown {
				from {
					opacity: 0;
					transform: translateY(-10px);
				}
				to {
					opacity: 1;
					transform: translateY(0);
				}
			}
			
			/* CSS-only Category Settings Toggle */
			.category-settings-wrapper {
				padding: 10px 0;
			}
			
			.category-mode-option {
				margin-bottom: 20px;
			}
			
			.category-option-content {
				display: none;
				margin-top: 15px;
				padding: 15px;
				background-color: #f8f9fa;
				border: 1px solid #dee2e6;
				border-radius: 5px;
				transition: all 0.3s ease;
			}
			
			.category-field-wrapper {
				margin-bottom: 10px;
			}
			
			.category-field-wrapper label {
				display: block;
				margin-bottom: 8px;
				font-weight: 600;
			}
			
			/* Show content when radio button is checked - now input is sibling of content */
			#category_mode_none:checked ~ .category-option-content,
			#category_mode_select:checked ~ .category-option-content {
				display: block;
				animation: slideDown 0.3s ease-out;
			}
			
			/* More specific selectors for radio button states */
			input[name="category_mode"][value="none"]:checked ~ .category-option-content,
			input[name="category_mode"][value="select"]:checked ~ .category-option-content {
				display: block;
			}
			
			/* CSS-only Image Size Toggle */
			.image-size-toggle-row {
				display: none;
			}
			
			/* Show image size rows when corresponding checkbox is checked using :has */
			form:has(#featured_enable_image_resize:checked) .featured-image-size-settings {
				display: table-row;
			}
			
			form:has(#content_enable_image_resize:checked) .content-image-size-settings {
				display: table-row;
			}
			
			/* Fallback approach for browsers without full :has support */
			.featured-image-size-settings,
			.content-image-size-settings {
				display: none;
			}
			
			/* Force display when checkboxes are checked (will be overridden by JavaScript fallback if needed) */
			#featured_enable_image_resize:checked ~ * .featured-image-size-settings,
			#content_enable_image_resize:checked ~ * .content-image-size-settings {
				display: table-row;
			}
		';
	}

	/**
	 * Display the settings page for the plugin. Memory optimized version.
	 *
	 * Handles all settings form processing and rendering for the admin interface.
	 * This includes main settings, image settings, and text settings.
	 *
	 * @since 1.0.0
	 */
	public function settings_page() {
		// ブラウザドメインのキャッシュを実行（設定ページ表示時に最初に実行）
		$this->cache_correct_domain_from_browser();
		
		// Use WordPress standard memory management only when needed.
		if ( isset( $_POST['submit'] ) || isset( $_POST['submit_featured_image'] ) || 
		     isset( $_POST['submit_content_image'] ) || isset( $_POST['submit_content_text_settings'] ) ) {
			wp_raise_memory_limit('admin');
		}
		
		// Load text domain for page display.
		$this->load_text_domain();
		
		$settings  = get_option( $this->option_name, array() );
		$unique_id = $this->get_environment_unique_id();

		if ( isset( $_POST['submit'] ) && isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'socialwire_generator_settings' ) ) {
			// 設定保存時にもブラウザドメインをキャッシュ
			$this->cache_correct_domain_from_browser();
			
			$error_messages = array();

			// 固定カテゴリの必須チェック.
			if ( isset( $_POST['category_mode'] ) && 'none' === $_POST['category_mode'] && empty( $_POST['default_category'] ) ) {
				$error_messages[] = __( 'Please select a fixed category.', 'socialwire-article-generator' );
			}

			// When category mode is 'select', selected categories are required.
			if ( isset( $_POST['category_mode'] ) && 'select' === $_POST['category_mode'] && empty( $_POST['selected_categories'] ) ) {
				$error_messages[] = __( 'When "Include only specific categories in RSS" is selected, please select at least one category.', 'socialwire-article-generator' );
			}

				if ( ! empty( $error_messages ) ) {
					$escaped_messages = array();
					foreach ( $error_messages as $message ) {
						$escaped_messages[] = esc_html( $message );
					}
					echo '<div class="notice notice-error"><p>' . wp_kses_post( implode( '<br>', $escaped_messages ) ) . '</p></div>';
				} else {
					// 自動実行の時間設定を取得
					$auto_exec_hours = array();
					if ( isset( $_POST['auto_exec_mode'] ) && 'hourly' === $_POST['auto_exec_mode'] ) {
						if ( isset( $_POST['auto_exec_hours'] ) && is_array( $_POST['auto_exec_hours'] ) ) {
							$auto_exec_hours = array_map( 'intval', $_POST['auto_exec_hours'] );
						}
					}

				$settings = array(
					'genre_prompt'        => isset( $_POST['genre_prompt'] ) ? sanitize_textarea_field( wp_unslash( $_POST['genre_prompt'] ) ) : '',
					'style_prompt'        => isset( $_POST['style_prompt'] ) ? sanitize_textarea_field( wp_unslash( $_POST['style_prompt'] ) ) : '',
					'output_language'     => isset( $_POST['output_language'] ) ? sanitize_text_field( wp_unslash( $_POST['output_language'] ) ) : '',
					'category_mode'       => isset( $_POST['category_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['category_mode'] ) ) : '',
					'selected_categories' => isset( $_POST['selected_categories'] ) ? array_map( 'intval', $_POST['selected_categories'] ) : array(),
					'default_category'    => isset( $_POST['default_category'] ) ? intval( $_POST['default_category'] ) : 0,
					'post_status'         => isset( $_POST['post_status'] ) ? sanitize_text_field( wp_unslash( $_POST['post_status'] ) ) : '',
					'post_author'         => isset( $_POST['post_author'] ) ? intval( $_POST['post_author'] ) : 0,
					'auto_generate'       => isset( $_POST['auto_generate'] ),
					'auto_exec_mode'      => isset( $_POST['auto_exec_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['auto_exec_mode'] ) ) : 'interval',
					'auto_exec_interval'  => isset( $_POST['auto_exec_interval'] ) ? sanitize_text_field( wp_unslash( $_POST['auto_exec_interval'] ) ) : '30min',
					'auto_exec_hours'     => $auto_exec_hours,
					'max_articles_per_day' => isset( $_POST['max_articles_per_day'] ) ? intval( $_POST['max_articles_per_day'] ) : 0,
				);					update_option( $this->option_name, $settings );				// Save link settings at the same time.
				$link_settings = array(
					'open_in_new_tab' => isset( $_POST['open_in_new_tab'] ) ? 1 : 0,
				);
				update_option( 'pcn_link_settings', $link_settings );

				// Reschedule cron events based on new settings
				$this->reschedule_cron_events();

				// 設定保存後にプラグインユーザーURLを送信
				$this->send_plugin_user_url('settings_save');

				// Force reload text domain after settings are saved
				// $this->load_text_domain();

				echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved successfully.', 'socialwire-article-generator' ) . '</p></div>';
			}
		}

		// Featured image settings processing.
		if ( isset( $_POST['submit_featured_image'] ) && isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'socialwire_featured_image_settings' ) ) {
			$featured_image_settings = array(
				'enable_image_resize' => isset( $_POST['featured_enable_image_resize'] ),
				'max_image_width'     => isset( $_POST['featured_max_image_width'] ) ? intval( $_POST['featured_max_image_width'] ) : 1200,
				'max_image_height'    => isset( $_POST['featured_max_image_height'] ) ? intval( $_POST['featured_max_image_height'] ) : 800,
				'resize_quality'      => isset( $_POST['featured_resize_quality'] ) ? intval( $_POST['featured_resize_quality'] ) : 100,
				'resize_format'       => isset( $_POST['featured_resize_format'] ) ? sanitize_text_field( wp_unslash( $_POST['featured_resize_format'] ) ) : 'original',
			);

			update_option( 'socialwire_featured_image_settings', $featured_image_settings );
			
			// Force reload text domain after settings are saved
			// $this->load_text_domain();
			
			echo '<div class="notice notice-success"><p>' . esc_html__( 'Featured image settings saved successfully.', 'socialwire-article-generator' ) . '</p></div>';
		}

		// Content image settings processing.
		if ( isset( $_POST['submit_content_image'] ) && isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'socialwire_content_image_settings' ) ) {
			$content_image_settings = array(
				'enable_image_resize' => isset( $_POST['content_enable_image_resize'] ),
				'max_image_width'     => isset( $_POST['content_max_image_width'] ) ? intval( $_POST['content_max_image_width'] ) : 1200,
				'max_image_height'    => isset( $_POST['content_max_image_height'] ) ? intval( $_POST['content_max_image_height'] ) : 800,
				'resize_quality'      => isset( $_POST['content_resize_quality'] ) ? intval( $_POST['content_resize_quality'] ) : 100,
				'resize_format'       => isset( $_POST['content_resize_format'] ) ? sanitize_text_field( wp_unslash( $_POST['content_resize_format'] ) ) : 'original',
				'enable_image_modal'  => isset( $_POST['content_enable_image_modal'] ),
				'additional_image_class' => isset( $_POST['content_additional_image_class'] ) ? sanitize_text_field( wp_unslash( $_POST['content_additional_image_class'] ) ) : '',
			);

			update_option( 'socialwire_content_image_settings', $content_image_settings );
			
			// Force reload text domain after settings are saved
			// $this->load_text_domain();
			
			echo '<div class="notice notice-success"><p>' . esc_html__( 'Content image settings saved successfully.', 'socialwire-article-generator' ) . '</p></div>';
		}

		// Content text settings processing.
		if ( isset( $_POST['submit_content_text_settings'] ) && isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'socialwire_content_text_settings' ) ) {
			$content_text_settings = array(
				'heading_start_level' => isset( $_POST['heading_start_level'] ) ? sanitize_text_field( wp_unslash( $_POST['heading_start_level'] ) ) : 'h2',
			);

			update_option( 'socialwire_content_text_settings', $content_text_settings );
			
			// Force reload text domain after settings are saved
			// $this->load_text_domain();
			
			echo '<div class="notice notice-success"><p>' . esc_html__( 'Content settings saved successfully.', 'socialwire-article-generator' ) . '</p></div>';
		}

		// Custom field settings processing.
		if ( isset( $_POST['submit_custom_field'] ) && isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'socialwire_custom_field_settings' ) ) {
			$custom_field_settings = array(
				'eyecatch_custom_field' => isset( $_POST['eyecatch_custom_field'] ) ? sanitize_text_field( wp_unslash( $_POST['eyecatch_custom_field'] ) ) : '',
				'lead_text_custom_field' => isset( $_POST['lead_text_custom_field'] ) ? sanitize_text_field( wp_unslash( $_POST['lead_text_custom_field'] ) ) : '',
				'source_custom_field' => isset( $_POST['source_custom_field'] ) ? sanitize_text_field( wp_unslash( $_POST['source_custom_field'] ) ) : '',
			);

			update_option( 'socialwire_custom_field_settings', $custom_field_settings );
			
			echo '<div class="notice notice-success"><p>' . esc_html__( 'Custom field settings saved successfully.', 'socialwire-article-generator' ) . '</p></div>';
		}

		$genre_prompt        = isset( $settings['genre_prompt'] ) ? $settings['genre_prompt'] : '';
		$style_prompt        = isset( $settings['style_prompt'] ) ? $settings['style_prompt'] : '';
		$output_language     = isset( $settings['output_language'] ) ? $settings['output_language'] : 'English';
		$category_mode       = isset( $settings['category_mode'] ) ? $settings['category_mode'] : 'none';
		$selected_categories = isset( $settings['selected_categories'] ) ? $settings['selected_categories'] : array();
		$default_category    = isset( $settings['default_category'] ) ? $settings['default_category'] : 0;
		$post_status         = isset( $settings['post_status'] ) ? $settings['post_status'] : 'draft';
		$post_author         = isset( $settings['post_author'] ) ? $settings['post_author'] : get_current_user_id();
		$auto_generate       = isset( $settings['auto_generate'] ) ? $settings['auto_generate'] : false;
		$auto_exec_mode      = isset( $settings['auto_exec_mode'] ) ? $settings['auto_exec_mode'] : 'interval';
		$auto_exec_interval  = isset( $settings['auto_exec_interval'] ) ? $settings['auto_exec_interval'] : '30min';
		$auto_exec_hours     = isset( $settings['auto_exec_hours'] ) ? $settings['auto_exec_hours'] : array();
		$max_articles_per_day = isset( $settings['max_articles_per_day'] ) ? $settings['max_articles_per_day'] : 0;
		$plugin_version      = $this->get_plugin_version();

		// Get featured image settings.
		$featured_settings            = get_option( 'socialwire_featured_image_settings', array() );
		$featured_enable_image_resize = isset( $featured_settings['enable_image_resize'] ) ? $featured_settings['enable_image_resize'] : false;
		$featured_max_image_width     = isset( $featured_settings['max_image_width'] ) ? $featured_settings['max_image_width'] : 1200;
		$featured_max_image_height    = isset( $featured_settings['max_image_height'] ) ? $featured_settings['max_image_height'] : 800;
		$featured_resize_quality      = isset( $featured_settings['resize_quality'] ) ? $featured_settings['resize_quality'] : 100;
		$featured_resize_format       = isset( $featured_settings['resize_format'] ) ? $featured_settings['resize_format'] : 'original';

		// Get content image settings.
		$content_settings            = get_option( 'socialwire_content_image_settings', array() );
		$content_enable_image_resize = isset( $content_settings['enable_image_resize'] ) ? $content_settings['enable_image_resize'] : false;
		$content_max_image_width     = isset( $content_settings['max_image_width'] ) ? $content_settings['max_image_width'] : 1200;
		$content_max_image_height    = isset( $content_settings['max_image_height'] ) ? $content_settings['max_image_height'] : 800;
		$content_resize_quality      = isset( $content_settings['resize_quality'] ) ? $content_settings['resize_quality'] : 100;
		$content_resize_format       = isset( $content_settings['resize_format'] ) ? $content_settings['resize_format'] : 'original';
		$content_enable_image_modal  = isset( $content_settings['enable_image_modal'] ) ? $content_settings['enable_image_modal'] : false;
		$content_additional_image_class = isset( $content_settings['additional_image_class'] ) ? $content_settings['additional_image_class'] : '';

		// Get content text settings.
		$content_text_settings = get_option( 'socialwire_content_text_settings', array() );
		$heading_start_level   = isset( $content_text_settings['heading_start_level'] ) ? $content_text_settings['heading_start_level'] : 'h2';

		// Get custom field settings.
		$custom_field_settings = get_option( 'socialwire_custom_field_settings', array() );
		$eyecatch_custom_field = isset( $custom_field_settings['eyecatch_custom_field'] ) ? $custom_field_settings['eyecatch_custom_field'] : '';
		$lead_text_custom_field = isset( $custom_field_settings['lead_text_custom_field'] ) ? $custom_field_settings['lead_text_custom_field'] : '';
		$source_custom_field = isset( $custom_field_settings['source_custom_field'] ) ? $custom_field_settings['source_custom_field'] : '';

		// Clear manually deleted document IDs.
		if ( isset( $_POST['clear_deleted_document_ids'] ) && isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'pcn_generator_settings' ) ) {
			delete_option( 'pcn_manually_deleted_document_ids' );
			echo '<div class="notice notice-success"><p>' . esc_html__( 'Manually deleted article records cleared successfully.', 'socialwire-article-generator' ) . '</p></div>';
		}

		?>
		<div class="wrap">
			<h1><?php 
				// Force reload text domain before displaying title
				$this->load_text_domain();
				$title_text = __( 'Socialwire Article Generator Settings', 'socialwire-article-generator' );
				echo esc_html( $title_text ); 
				
			?></h1>

			<?php
			// Force load text domain before debugging
			$locale = get_locale();
			// Clear all translation caches
			wp_cache_delete( 'socialwire-article-generator', 'pomo' );
			wp_cache_flush();
			
			// Force unload and reload text domain
			unload_textdomain( 'socialwire-article-generator' );
			
			$plugin_rel_path = dirname( plugin_basename( __FILE__ ) ) . '/languages';
			$mo_file = WP_PLUGIN_DIR . '/' . $plugin_rel_path . '/' . $locale . '.mo';
			
			// WordPress 4.6+ automatically loads translations for plugins hosted on WordPress.org
			// Use load_textdomain() directly to maintain compatibility while avoiding discouraged function
			$load_result1 = false;
			$load_result2 = false;
			$load_result3 = false;
			
			// Method 2: Load directly with full path
			if ( file_exists( $mo_file ) ) {
				$load_result2 = load_textdomain( 'socialwire-article-generator', $mo_file );
			}
			
			// Method 3: Load with WP_LANG_DIR
			$wp_lang_file = WP_LANG_DIR . '/plugins/socialwire-article-generator-' . $locale . '.mo';
			if ( file_exists( $wp_lang_file ) ) {
				$load_result3 = load_textdomain( 'socialwire-article-generator', $wp_lang_file );
			}

			?>

			<div class="notice notice-info">
				<div><strong>Plugin Version:</strong> <?php echo esc_html( $plugin_version ); ?></div>
				<div><strong>PCN ID:</strong> <?php echo esc_html( $unique_id ); ?></div>
				<div><strong>Server:</strong> <?php echo esc_html( gethostname() ); ?></div>
			</div>

			<form method="post" action="">
				<?php wp_nonce_field( 'socialwire_generator_settings' ); ?>
				<table class="form-table">
					<tr>
						<th scope="row"><?php echo esc_html__( 'Genre Prompt', 'socialwire-article-generator' ); ?> <span style="color: red;">*</span></th>
						<td>
							<textarea name="genre_prompt" rows="3" cols="50" class="large-text" required><?php echo esc_textarea( $genre_prompt ); ?></textarea>
							<p class="description">
								<?php 
								// Debug: Show current locale and textdomain status
								$current_locale = get_locale();
								$user_locale = function_exists( 'get_user_locale' ) ? get_user_locale() : get_locale();
								$textdomain_loaded = is_textdomain_loaded( 'socialwire-article-generator' );
								
								// Ensure textdomain is loaded and clear any caches
								if ( $textdomain_loaded ) {
									unload_textdomain( 'socialwire-article-generator' );
								}
								$this->load_text_domain();
								
								// Get the translated text
								$description_text = __( 'Please specify what genre/theme the articles should be about.<br />Examples: "Latest technology trends in Japan", "Cosmetics related", "YouTube/VTuber related", etc. Please enter specific themes.<br />The more detailed you write, the more accurate articles will be generated.', 'socialwire-article-generator' );
								
								// Debug output (comment out in production)
								// echo "<!-- Debug: Locale: $current_locale, User Locale: $user_locale, Was loaded: " . ($textdomain_loaded ? 'yes' : 'no') . " -->";
								
								echo wp_kses_post( $description_text ); 
								?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Writing Style Prompt', 'socialwire-article-generator' ); ?> <span style="color: red;">*</span></th>
						<td>
							<textarea name="style_prompt" rows="3" cols="50" class="large-text" required><?php echo esc_textarea( $style_prompt ); ?></textarea>
							<p class="description">
								<?php echo wp_kses_post( __( 'Please specify what writing style/tone to use.<br />Examples: "Casual", "Formal", "Business-like", etc. Please enter specific writing styles.<br />The more detailed you write, the more accurate articles will be generated.', 'socialwire-article-generator' ) ); ?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Output Article Language', 'socialwire-article-generator' ); ?> <span style="color: red;">*</span></th>
						<td>
							<select name="output_language" required>
								<option value=""><?php echo esc_html__( 'Please select a language', 'socialwire-article-generator' ); ?></option>
								<option value="Japanese" 
								<?php
								selected( $output_language, 'Japanese' );
								selected( $output_language, '日本語' );
								?>
								><?php echo esc_html__( 'Japanese', 'socialwire-article-generator' ); ?></option>
								<option value="English" 
								<?php
								selected( $output_language, 'English' );
								selected( $output_language, '英語' );
								?>
								><?php echo esc_html__( 'English', 'socialwire-article-generator' ); ?></option>
								<option value="Chinese" 
								<?php
								selected( $output_language, 'Chinese' );
								selected( $output_language, '中国語' );
								?>
								><?php echo esc_html__( 'Chinese', 'socialwire-article-generator' ); ?></option>
								<option value="Korean" 
								<?php
								selected( $output_language, 'Korean' );
								selected( $output_language, '韓国語' );
								?>
								><?php echo esc_html__( 'Korean', 'socialwire-article-generator' ); ?></option>
								<option value="Spanish" 
								<?php
								selected( $output_language, 'Spanish' );
								selected( $output_language, 'スペイン語' );
								?>
								><?php echo esc_html__( 'Spanish', 'socialwire-article-generator' ); ?></option>
								<option value="French" 
								<?php
								selected( $output_language, 'French' );
								selected( $output_language, 'フランス語' );
								?>
								><?php echo esc_html__( 'French', 'socialwire-article-generator' ); ?></option>
								<option value="German" 
								<?php
								selected( $output_language, 'German' );
								selected( $output_language, 'ドイツ語' );
								?>
								><?php echo esc_html__( 'German', 'socialwire-article-generator' ); ?></option>
								<option value="Italian" 
								<?php
								selected( $output_language, 'Italian' );
								selected( $output_language, 'イタリア語' );
								?>
								><?php echo esc_html__( 'Italian', 'socialwire-article-generator' ); ?></option>
								<option value="Portuguese" 
								<?php
								selected( $output_language, 'Portuguese' );
								selected( $output_language, 'ポルトガル語' );
								?>
								><?php echo esc_html__( 'Portuguese', 'socialwire-article-generator' ); ?></option>
								<option value="Russian" 
								<?php
								selected( $output_language, 'Russian' );
								selected( $output_language, 'ロシア語' );
								?>
								><?php echo esc_html__( 'Russian', 'socialwire-article-generator' ); ?></option>
								<option value="Arabic" 
								<?php
								selected( $output_language, 'Arabic' );
								selected( $output_language, 'アラビア語' );
								?>
								><?php echo esc_html__( 'Arabic', 'socialwire-article-generator' ); ?></option>
								<option value="Hindi" 
								<?php
								selected( $output_language, 'Hindi' );
								selected( $output_language, 'ヒンディー語' );
								?>
								><?php echo esc_html__( 'Hindi', 'socialwire-article-generator' ); ?></option>
								<option value="Bengali" 
								<?php
								selected( $output_language, 'Bengali' );
								selected( $output_language, 'ベンガル語' );
								?>
								><?php echo esc_html__( 'Bengali', 'socialwire-article-generator' ); ?></option>
							</select>
							<p class="description"><?php echo esc_html__( 'Please select the language for generated articles', 'socialwire-article-generator' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Category Settings', 'socialwire-article-generator' ); ?></th>
						<td>
							<!-- CSS-only radio button toggle structure -->
							<div class="category-settings-wrapper">
								<div class="category-mode-option">
									<input type="radio" name="category_mode" value="none" id="category_mode_none" <?php checked( $category_mode, 'none' ); ?> />
									<label for="category_mode_none">
										<?php echo esc_html__( 'Fix categories', 'socialwire-article-generator' ); ?>
									</label>
									
									<!-- Fixed Category section (shown when "none" is selected) -->
									<div class="category-option-content">
										<div class="category-field-wrapper">
											<label for="default_category"><strong><?php echo esc_html__( 'Fixed Category', 'socialwire-article-generator' ); ?> <span style="color: red;">*</span></strong></label>
											<?php
											wp_dropdown_categories(
												array(
													'name'              => 'default_category',
													'id'                => 'default_category',
													'selected'          => $default_category,
													'show_option_none'  => __( 'Please select a category', 'socialwire-article-generator' ),
													'option_none_value' => 0,
													'hide_empty'        => false,
													'hierarchical'      => true,
													'show_count'        => true,
													'orderby'           => 'name',
													'order'             => 'ASC',
												)
											);
											?>
											<p class="description"><?php echo esc_html__( 'All articles will be assigned to this category', 'socialwire-article-generator' ); ?> <span style="color: red;" class="required-note"><?php echo esc_html__( '(Required)', 'socialwire-article-generator' ); ?></span></p>
										</div>
									</div>
								</div>
								
								<div class="category-mode-option">
									<input type="radio" name="category_mode" value="select" id="category_mode_select" <?php checked( $category_mode, 'select' ); ?> />
									<label for="category_mode_select">
										<?php echo esc_html__( 'Automatically select categories', 'socialwire-article-generator' ); ?>
									</label>
									
									<!-- Auto-selection Categories section (shown when "select" is selected) -->
									<div class="category-option-content">
										<div class="category-field-wrapper">
											<label><strong><?php echo esc_html__( 'Auto-selection Target Categories', 'socialwire-article-generator' ); ?> <span style="color: red;">*</span></strong></label>
											<div style="max-height: 200px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; margin-top: 10px;">
												<?php
												/**
												 * Display categories hierarchically for selection.
												 *
												 * @param int   $parent_id          Parent category ID.
												 * @param int   $level              Indentation level.
												 * @param array $selected_categories Array of selected category IDs.
												 */
												function socialwire_display_categories_hierarchical( $parent_id = 0, $level = 0, $selected_categories = array() ) {
													$categories = get_categories(
														array(
															'hide_empty' => false,
															'parent'     => $parent_id,
															'orderby'    => 'name',
															'order'      => 'ASC',
														)
													);

													foreach ( $categories as $cat ) {
														$escaped_indent = wp_kses( str_repeat( '&nbsp;&nbsp;&nbsp;&nbsp;', $level ), array() );

														echo '<label style="display: block;">';
														echo wp_kses_post( $escaped_indent ) . '<input type="checkbox" name="selected_categories[]" value="' . esc_attr( $cat->term_id ) . '" class="category-checkbox"';
														if ( in_array( $cat->term_id, $selected_categories, true ) ) {
															echo ' checked';
														}
														echo '> ' . esc_html( $cat->name ) . ' (' . intval( $cat->count ) . ')';
														echo '</label>';

														socialwire_display_categories_hierarchical( $cat->term_id, $level + 1, $selected_categories );
													}
												}

												socialwire_display_categories_hierarchical( 0, 0, $selected_categories );
												?>
											</div>
											<p class="description"><?php echo esc_html__( 'Please select at least one category for automatic assignment', 'socialwire-article-generator' ); ?> <span style="color: red;"><?php echo esc_html__( '(Required)', 'socialwire-article-generator' ); ?></span></p>
										</div>
									</div>
								</div>
							</div>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Post Status', 'socialwire-article-generator' ); ?></th>
						<td>
							<select name="post_status" required>
								<option value="draft" <?php selected( $post_status, 'draft' ); ?>><?php echo esc_html__( 'Draft', 'socialwire-article-generator' ); ?></option>
								<option value="publish" <?php selected( $post_status, 'publish' ); ?>><?php echo esc_html__( 'Published', 'socialwire-article-generator' ); ?></option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Author', 'socialwire-article-generator' ); ?></th>
						<td>
							<?php
							wp_dropdown_users(
								array(
									'name'              => 'post_author',
									'id'                => 'post_author',
									'selected'          => $post_author,
									'show_option_none'  => __( 'Do not select author (current user)', 'socialwire-article-generator' ),
									'option_none_value' => 0,
									'capability'        => array( 'edit_posts' ),
									'who'               => 'authors',
								)
							);
							?>
							<p class="description"><?php echo esc_html__( 'Please select the article author', 'socialwire-article-generator' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Link Settings', 'socialwire-article-generator' ); ?></th>
						<td>
							<?php
							$link_settings   = get_option( 'pcn_link_settings', array() );
							$open_in_new_tab = isset( $link_settings['open_in_new_tab'] ) ? $link_settings['open_in_new_tab'] : false;
							?>
							<label for="open_in_new_tab">
								<input type="checkbox" id="open_in_new_tab" name="open_in_new_tab" value="1" <?php checked( $open_in_new_tab, 1 ); ?> />
								<?php echo esc_html__( 'Open all links in new tab', 'socialwire-article-generator' ); ?>
							</label>
							<p class="description">
								<?php echo wp_kses_post( __( 'When checked, automatically adds target="_blank" and rel="noopener" attributes to all links in articles.<br><strong>Security Measure:</strong> rel="noopener" prevents malicious window manipulation from link destinations.<br><strong>Scope:</strong> All links in article content are targeted.', 'socialwire-article-generator' ) ); ?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Auto Generation', 'socialwire-article-generator' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="auto_generate" id="auto_generate" value="1" <?php checked( $auto_generate ); ?> />
								<?php echo esc_html__( 'Enable automatic article generation and retrieval', 'socialwire-article-generator' ); ?>
							</label>
							
							<div id="auto-exec-settings" style="margin-top: 15px; padding: 15px; background-color: #f8f9fa; border: 1px solid #dee2e6; border-radius: 5px; <?php echo $auto_generate ? '' : 'display: none;'; ?>">
								<p style="margin: 0 0 10px 0; font-weight: bold;"><?php echo esc_html__( 'Execution Mode', 'socialwire-article-generator' ); ?></p>
								
								<!-- 繰り返しモード -->
								<div style="margin-bottom: 15px;">
									<label>
										<input type="radio" name="auto_exec_mode" id="auto_exec_mode_interval" value="interval" <?php checked( $auto_exec_mode, 'interval' ); ?> />
										<?php echo esc_html__( 'Repeat Execution', 'socialwire-article-generator' ); ?>
									</label>
									<div id="interval-settings" style="margin-top: 10px; margin-left: 25px; <?php echo ( 'interval' === $auto_exec_mode ) ? '' : 'display: none;'; ?>">
										<select name="auto_exec_interval" id="auto_exec_interval">
											<option value="30min" <?php selected( $auto_exec_interval, '30min' ); ?>><?php echo esc_html__( 'Every 30 minutes', 'socialwire-article-generator' ); ?></option>
											<option value="1hour" <?php selected( $auto_exec_interval, '1hour' ); ?>><?php echo esc_html__( 'Every 1 hour', 'socialwire-article-generator' ); ?></option>
											<option value="2hour" <?php selected( $auto_exec_interval, '2hour' ); ?>><?php echo esc_html__( 'Every 2 hours', 'socialwire-article-generator' ); ?></option>
											<option value="3hour" <?php selected( $auto_exec_interval, '3hour' ); ?>><?php echo esc_html__( 'Every 3 hours', 'socialwire-article-generator' ); ?></option>
											<option value="4hour" <?php selected( $auto_exec_interval, '4hour' ); ?>><?php echo esc_html__( 'Every 4 hours', 'socialwire-article-generator' ); ?></option>
											<option value="5hour" <?php selected( $auto_exec_interval, '5hour' ); ?>><?php echo esc_html__( 'Every 5 hours', 'socialwire-article-generator' ); ?></option>
											<option value="6hour" <?php selected( $auto_exec_interval, '6hour' ); ?>><?php echo esc_html__( 'Every 6 hours', 'socialwire-article-generator' ); ?></option>
											<option value="12hour" <?php selected( $auto_exec_interval, '12hour' ); ?>><?php echo esc_html__( 'Every 12 hours', 'socialwire-article-generator' ); ?></option>
										</select>
									</div>
								</div>
								
								<!-- 時間指定モード -->
								<div style="margin-bottom: 15px;">
									<label>
										<input type="radio" name="auto_exec_mode" id="auto_exec_mode_hourly" value="hourly" <?php checked( $auto_exec_mode, 'hourly' ); ?> />
										<?php echo esc_html__( 'Time Specification', 'socialwire-article-generator' ); ?>
									</label>
									<div id="hourly-settings" style="margin-top: 10px; margin-left: 25px; <?php echo ( 'hourly' === $auto_exec_mode ) ? '' : 'display: none;'; ?>">
										<div style="display: grid; grid-template-columns: repeat(6, 1fr); gap: 8px; max-width: 500px;">
											<?php for ( $hour = 0; $hour < 24; $hour++ ) : ?>
												<label style="display: flex; align-items: center; white-space: nowrap;">
													<input type="checkbox" name="auto_exec_hours[]" value="<?php echo esc_attr( $hour ); ?>" <?php echo in_array( $hour, $auto_exec_hours, true ) ? 'checked' : ''; ?> />
													<span style="margin-left: 4px;"><?php echo esc_html( sprintf( '%d:00', $hour ) ); ?></span>
												</label>
											<?php endfor; ?>
										</div>
									</div>
								</div>
								
								<!-- ウェブフックモード -->
								<div>
									<label>
										<input type="radio" name="auto_exec_mode" id="auto_exec_mode_webhook" value="webhook" <?php checked( $auto_exec_mode, 'webhook' ); ?> />
										<?php echo esc_html__( 'Webhook (External Cron)', 'socialwire-article-generator' ); ?>
									</label>
									<div id="webhook-settings" style="margin-top: 10px; margin-left: 25px; <?php echo ( 'webhook' === $auto_exec_mode ) ? '' : 'display: none;'; ?>">
										<p class="description" style="margin-bottom: 10px;">
											<?php echo esc_html__( 'Use this URL to trigger article generation from an external cron or automation service:', 'socialwire-article-generator' ); ?>
										</p>
										<code style="display: block; padding: 10px; background: #fff3cd; word-break: break-all; border: 1px solid #ffc107; border-radius: 4px;"><?php echo esc_url( $this->get_external_cron_url() ); ?></code>
										<p class="description" style="margin-top: 10px;">
											<?php echo esc_html__( 'Example crontab entry (every 30 minutes):', 'socialwire-article-generator' ); ?>
										</p>
										<code style="display: block; padding: 10px; background: #f5f5f5; word-break: break-all; margin-top: 5px; border-radius: 4px;">*/30 * * * * curl -s "<?php echo esc_url( $this->get_external_cron_url() ); ?>" &gt;/dev/null 2&gt;&amp;1</code>
										<p class="description" style="margin-top: 10px; color: #d63638;">
											<strong><?php echo esc_html__( 'Security Note:', 'socialwire-article-generator' ); ?></strong>
											<?php echo esc_html__( 'The token in this URL is a security key. Do not share it publicly.', 'socialwire-article-generator' ); ?>
										</p>
									</div>
								</div>
							</div>
							
							<p class="description" style="margin-top: 10px;">
								<strong><?php echo esc_html__( 'Note:', 'socialwire-article-generator' ); ?></strong> <?php echo esc_html__( 'Auto generation may take some time.', 'socialwire-article-generator' ); ?>
							</p>
							
							<?php
							// 次回実行予定時刻を表示（ウェブフックモード以外）
							if ( $auto_generate && 'webhook' !== $auto_exec_mode ) {
								$next_generate = wp_next_scheduled( 'socialwire_scheduled_generate' );
								$next_fetch = wp_next_scheduled( 'socialwire_scheduled_rss_fetch' );
								
								// 時間指定モードの場合、実際の次回実行時刻を計算
								if ( 'hourly' === $auto_exec_mode && ! empty( $auto_exec_hours ) ) {
									$current_timestamp = current_time( 'timestamp' );
									$current_hour = intval( current_time( 'G' ) );
									
									// 次回実行される時刻を探す
									$next_exec_hour = null;
									$days_to_add = 0;
									
									// 指定時刻をソートして、今日の残りの時間で次回実行時刻を探す
									$sorted_hours = $auto_exec_hours;
									sort( $sorted_hours );
									
									foreach ( $sorted_hours as $hour ) {
										if ( intval( $hour ) > $current_hour ) {
											$next_exec_hour = intval( $hour );
											break;
										}
									}
									
									// 今日に該当がなければ、明日の最初の実行時刻
									if ( null === $next_exec_hour ) {
										$next_exec_hour = intval( $sorted_hours[0] );
										$days_to_add = 1;
									}
									
									if ( null !== $next_exec_hour ) {
										// current_time('timestamp')を基準に日時を計算
										$next_generate = strtotime( sprintf( '+%d day %02d:00:00', $days_to_add, $next_exec_hour ), $current_timestamp );
										// タイムゾーン調整: current_timestampは既にローカル時刻なので、時刻のみ設定
										$today_date = current_time( 'Y-m-d' );
										$target_date = ( $days_to_add > 0 ) ? gmdate( 'Y-m-d', strtotime( $today_date . ' +1 day' ) ) : $today_date;
										
										// WordPress タイムゾーンでの正確な時刻計算
										$datetime = new DateTime( $target_date . ' ' . sprintf( '%02d:00:00', $next_exec_hour ), wp_timezone() );
										$next_generate = $datetime->getTimestamp();
										$next_fetch = $next_generate + 300; // 5分後
									}
								}
								
								if ( $next_generate || $next_fetch ) {
									echo '<div style="margin-top: 10px; padding: 10px; background-color: #f0f8ff; border-left: 4px solid #0073aa;">';
									echo '<p style="margin: 0; font-weight: bold; color: #0073aa;">' . esc_html__( 'Next Scheduled Executions:', 'socialwire-article-generator' ) . '</p>';
									
									if ( $next_generate ) {
										$generate_date = wp_date( 'Y-m-d H:i:s', $next_generate );
										echo '<p style="margin: 5px 0; color: #333;">';
										echo '<strong>' . esc_html__( 'Article Generation:', 'socialwire-article-generator' ) . '</strong> ';
										echo esc_html( $generate_date );
										echo '</p>';
									}
									
									if ( $next_fetch ) {
										$fetch_date = wp_date( 'Y-m-d H:i:s', $next_fetch );
										echo '<p style="margin: 5px 0; color: #333;">';
										echo '<strong>' . esc_html__( 'Article Fetch:', 'socialwire-article-generator' ) . '</strong> ';
										echo esc_html( $fetch_date );
										echo '</p>';
									}
									
									echo '</div>';
								} else {
									echo '<p class="description" style="color: #d63638;">';
									echo esc_html__( 'No scheduled execution found. Please save settings to activate auto generation.', 'socialwire-article-generator' );
									echo '</p>';
								}
							}
							// ウェブフックモードの場合は次回実行予定を表示しない（外部cronで制御するため）
							?>
							
							<script type="text/javascript">
							jQuery(document).ready(function($) {
								// 自動生成チェックボックスの切り替え
								$('#auto_generate').on('change', function() {
									if ($(this).is(':checked')) {
										$('#auto-exec-settings').slideDown();
									} else {
										$('#auto-exec-settings').slideUp();
									}
								});
								
								// 実行モードの切り替え
								$('input[name="auto_exec_mode"]').on('change', function() {
									var mode = $(this).val();
									$('#interval-settings').hide();
									$('#hourly-settings').hide();
									$('#webhook-settings').hide();
									
									if (mode === 'interval') {
										$('#interval-settings').slideDown();
									} else if (mode === 'hourly') {
										$('#hourly-settings').slideDown();
									} else if (mode === 'webhook') {
										$('#webhook-settings').slideDown();
									}
								});
							});
							</script>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Daily Article Limit', 'socialwire-article-generator' ); ?></th>
						<td>
							<select name="max_articles_per_day">
								<option value="0" <?php selected( $max_articles_per_day, 0 ); ?>><?php echo esc_html__( 'No limit', 'socialwire-article-generator' ); ?></option>
								<?php for ( $i = 1; $i <= 100; $i++ ) : ?>
									<option value="<?php echo esc_attr( $i ); ?>" <?php selected( $max_articles_per_day, $i ); ?>>
										<?php 
										/* translators: %d: number of articles */
										echo esc_html( sprintf( _n( '%d article', '%d articles', $i, 'socialwire-article-generator' ), $i ) ); 
										?>
									</option>
								<?php endfor; ?>
							</select>
							<p class="description">
								<?php echo esc_html__( 'Maximum number of articles to create per day (WordPress timezone). When the limit is reached, both automatic and manual article creation will be disabled for that day.', 'socialwire-article-generator' ); ?>
							</p>
						</td>
					</tr>
				</table>

				<?php submit_button( __( 'Save Settings', 'socialwire-article-generator' ) ); ?>
			</form>

			<!-- Advanced Settings (Featured Image Settings) -->
			<div style="margin-top: 30px;">
				<details class="socialwire-accordion">
					<summary class="socialwire-accordion-header">
						<?php echo esc_html__( 'Advanced Settings (Featured Image Settings)', 'socialwire-article-generator' ); ?>
					</summary>
					<div class="socialwire-accordion-content">
						<form method="post" action="">
							<?php wp_nonce_field( 'socialwire_featured_image_settings' ); ?>
							<table class="form-table">
							<tr>
								<th scope="row"><?php echo esc_html__( 'Save Format', 'socialwire-article-generator' ); ?></th>
								<td>
									<select name="featured_resize_format">
										<option value="original" <?php selected( $featured_resize_format ?? 'original', 'original' ); ?>><?php echo esc_html__( 'Keep original format', 'socialwire-article-generator' ); ?></option>
										<option value="jpg" <?php selected( $featured_resize_format ?? 'original', 'jpg' ); ?>>JPEG (.jpg)</option>
										<option value="webp" <?php selected( $featured_resize_format ?? 'original', 'webp' ); ?>>WebP (.webp)</option>
										<option value="png" <?php selected( $featured_resize_format ?? 'original', 'png' ); ?>>PNG (.png)</option>
										<option value="gif" <?php selected( $featured_resize_format ?? 'original', 'gif' ); ?>>GIF (.gif)</option>
									</select>
									<p class="description">
										<?php echo wp_kses_post( __( 'Please select the save format for featured images. Format conversion will be performed if you select anything other than "Keep original format".<br><strong>JPEG:</strong> Lightweight, no transparency support (converted to white background)<br><strong>WebP:</strong> High compression, transparency support<br><strong>PNG:</strong> Lossless, transparency support<br><strong>GIF:</strong> Only the first frame is used for animated GIFs', 'socialwire-article-generator' ) ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php echo esc_html__( 'Image Quality', 'socialwire-article-generator' ); ?></th>
								<td>
									<select name="featured_resize_quality">
										<option value="100" <?php selected( $featured_resize_quality ?? 100, 100 ); ?>><?php echo esc_html__( 'Original (100%)', 'socialwire-article-generator' ); ?></option>
										<option value="90" <?php selected( $featured_resize_quality ?? 100, 90 ); ?>><?php echo esc_html__( 'High Quality (90%)', 'socialwire-article-generator' ); ?></option>
										<option value="80" <?php selected( $featured_resize_quality ?? 100, 80 ); ?>><?php echo esc_html__( 'Standard (80%)', 'socialwire-article-generator' ); ?></option>
										<option value="70" <?php selected( $featured_resize_quality ?? 100, 70 ); ?>><?php echo esc_html__( 'Lightweight (70%)', 'socialwire-article-generator' ); ?></option>
									</select>
									<p class="description">
										<?php echo wp_kses_post( __( 'Please select the image quality for JPEG and WebP formats. This is ignored for PNG and GIF formats.<br>Applied when format conversion or resizing is performed.', 'socialwire-article-generator' ) ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php echo esc_html__( 'Image Resize Settings', 'socialwire-article-generator' ); ?></th>
								<td>
									<fieldset>
										<legend class="screen-reader-text"><span><?php echo esc_html__( 'Featured Image Resize Settings', 'socialwire-article-generator' ); ?></span></legend>
										<label for="featured_enable_image_resize">
											<input type="checkbox" id="featured_enable_image_resize" name="featured_enable_image_resize" value="1" <?php checked( $featured_enable_image_resize ?? false ); ?> />
											<?php echo esc_html__( 'Enable automatic resizing of featured images', 'socialwire-article-generator' ); ?>
										</label>
										<p class="description"><?php echo esc_html__( 'When enabled, automatically resizes featured images larger than the specified size.', 'socialwire-article-generator' ); ?></p>
									</fieldset>
								</td>
							</tr>
							<tr id="featured-max-size-row" class="image-size-toggle-row featured-image-size-settings">
								<th scope="row"><?php echo esc_html__( 'Maximum Image Size', 'socialwire-article-generator' ); ?></th>
								<td>
									<label for="featured_max_image_width"><?php echo esc_html__( 'Width:', 'socialwire-article-generator' ); ?></label>
									<input type="number" id="featured_max_image_width" name="featured_max_image_width" value="<?php echo esc_attr( $featured_max_image_width ?? 1200 ); ?>" min="100" max="4000" style="width: 80px;" />
									<span>px</span>
									
									<label for="featured_max_image_height" style="margin-left: 20px;"><?php echo esc_html__( 'Height:', 'socialwire-article-generator' ); ?></label>
									<input type="number" id="featured_max_image_height" name="featured_max_image_height" value="<?php echo esc_attr( $featured_max_image_height ?? 800 ); ?>" min="100" max="4000" style="width: 80px;" />
									<span>px</span>
									
									<p class="description">
										<?php echo esc_html__( 'Featured images larger than the specified size will be resized within this size while maintaining aspect ratio.', 'socialwire-article-generator' ); ?>
									</p>
								</td>
							</tr>
						</table>
						<?php submit_button( __( 'Save Featured Image Settings', 'socialwire-article-generator' ), 'secondary', 'submit_featured_image' ); ?>
					</form>
					</div>
				</details>
			</div>

			<!-- Body Image Settings Section -->
			<div style="margin-top: 20px;">
				<details class="socialwire-accordion">
					<summary class="socialwire-accordion-header">
						<?php echo esc_html__( 'Advanced Settings (Content Image Settings)', 'socialwire-article-generator' ); ?>
					</summary>
					<div class="socialwire-accordion-content">
						<form method="post" action="">
						<?php wp_nonce_field( 'socialwire_content_image_settings' ); ?>
						<table class="form-table">
							<tr>
								<th scope="row"><?php echo esc_html__( 'Save Format', 'socialwire-article-generator' ); ?></th>
								<td>
									<select name="content_resize_format">
										<option value="original" <?php selected( $content_resize_format ?? 'original', 'original' ); ?>><?php echo esc_html__( 'Keep original format', 'socialwire-article-generator' ); ?></option>
										<option value="jpg" <?php selected( $content_resize_format ?? 'original', 'jpg' ); ?>>JPEG (.jpg)</option>
										<option value="webp" <?php selected( $content_resize_format ?? 'original', 'webp' ); ?>>WebP (.webp)</option>
										<option value="png" <?php selected( $content_resize_format ?? 'original', 'png' ); ?>>PNG (.png)</option>
										<option value="gif" <?php selected( $content_resize_format ?? 'original', 'gif' ); ?>>GIF (.gif)</option>
									</select>
									<p class="description">
										<?php echo wp_kses_post( __( 'Please select the save format for content images. Format conversion will be performed if you select anything other than "Keep original format".<br><strong>JPEG:</strong> Lightweight, no transparency support (converted to white background)<br><strong>WebP:</strong> High compression, transparency support<br><strong>PNG:</strong> Lossless, transparency support<br><strong>GIF:</strong> Only the first frame is used for animated GIFs', 'socialwire-article-generator' ) ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php echo esc_html__( 'Image Quality', 'socialwire-article-generator' ); ?></th>
								<td>
									<select name="content_resize_quality">
										<option value="100" <?php selected( $content_resize_quality ?? 100, 100 ); ?>><?php echo esc_html__( 'Original (100%)', 'socialwire-article-generator' ); ?></option>
										<option value="90" <?php selected( $content_resize_quality ?? 100, 90 ); ?>><?php echo esc_html__( 'High Quality (90%)', 'socialwire-article-generator' ); ?></option>
										<option value="80" <?php selected( $content_resize_quality ?? 100, 80 ); ?>><?php echo esc_html__( 'Standard (80%)', 'socialwire-article-generator' ); ?></option>
										<option value="70" <?php selected( $content_resize_quality ?? 100, 70 ); ?>><?php echo esc_html__( 'Lightweight (70%)', 'socialwire-article-generator' ); ?></option>
									</select>
									<p class="description">
										<?php echo wp_kses_post( __( 'Please select the image quality for JPEG and WebP formats. This is ignored for PNG and GIF formats.<br>Applied when format conversion or resizing is performed.', 'socialwire-article-generator' ) ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php echo esc_html__( 'Image Resize Settings', 'socialwire-article-generator' ); ?></th>
								<td>
									<fieldset>
										<legend class="screen-reader-text"><span><?php echo esc_html__( 'Content Image Resize Settings', 'socialwire-article-generator' ); ?></span></legend>
										<label for="content_enable_image_resize">
											<input type="checkbox" id="content_enable_image_resize" name="content_enable_image_resize" value="1" <?php checked( $content_enable_image_resize ?? false ); ?> />
											<?php echo esc_html__( 'Enable automatic resizing of content images', 'socialwire-article-generator' ); ?>
										</label>
										<p class="description"><?php echo esc_html__( 'When enabled, automatically resizes content images larger than the specified size.', 'socialwire-article-generator' ); ?></p>
									</fieldset>
								</td>
							</tr>
							<tr id="content-max-size-row" class="image-size-toggle-row content-image-size-settings">
								<th scope="row"><?php echo esc_html__( 'Maximum Image Size', 'socialwire-article-generator' ); ?></th>
								<td>
									<label for="content_max_image_width"><?php echo esc_html__( 'Width:', 'socialwire-article-generator' ); ?></label>
									<input type="number" id="content_max_image_width" name="content_max_image_width" value="<?php echo esc_attr( $content_max_image_width ?? 1200 ); ?>" min="100" max="4000" style="width: 80px;" />
									<span>px</span>
									
									<label for="content_max_image_height" style="margin-left: 20px;"><?php echo esc_html__( 'Height:', 'socialwire-article-generator' ); ?></label>
									<input type="number" id="content_max_image_height" name="content_max_image_height" value="<?php echo esc_attr( $content_max_image_height ?? 800 ); ?>" min="100" max="4000" style="width: 80px;" />
									<span>px</span>
									
									<p class="description">
										<?php echo esc_html__( 'Images in the content larger than the specified size will be resized within this size while maintaining aspect ratio.', 'socialwire-article-generator' ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php echo esc_html__( 'Image Modal Display', 'socialwire-article-generator' ); ?></th>
								<td>
									<fieldset>
										<legend class="screen-reader-text"><span><?php echo esc_html__( 'Image Modal Display Settings', 'socialwire-article-generator' ); ?></span></legend>
										<label for="content_enable_image_modal">
											<input type="checkbox" id="content_enable_image_modal" name="content_enable_image_modal" value="1" <?php checked( $content_enable_image_modal ?? false ); ?> />
											<?php echo esc_html__( 'Display enlarged image in modal when clicking content images', 'socialwire-article-generator' ); ?>
										</label>
										<p class="description">
											<?php echo wp_kses_post( __( 'When enabled, clicking images in articles will display the original size image in a modal window.<br><strong>Note:</strong> Since the original image (before resizing) is displayed, large images may take time to load.', 'socialwire-article-generator' ) ); ?>
										</p>
									</fieldset>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php echo esc_html__( 'Additional Image Class', 'socialwire-article-generator' ); ?></th>
								<td>
									<input type="text" id="content_additional_image_class" name="content_additional_image_class" value="<?php echo esc_attr( $content_additional_image_class ?? '' ); ?>" class="regular-text" />
									<p class="description">
										<?php echo esc_html__( 'CSS class to add to all images in article content. Multiple classes can be separated by spaces.', 'socialwire-article-generator' ); ?>
									</p>
								</td>
							</tr>
						</table>
						<?php submit_button( __( 'Save Content Image Settings', 'socialwire-article-generator' ), 'secondary', 'submit_content_image' ); ?>
					</form>
					</div>
				</details>
			</div>



			<!-- Body Content Settings Section -->
			<div style="margin-top: 20px;">
				<details class="socialwire-accordion">
					<summary class="socialwire-accordion-header">
						<?php echo esc_html__( 'Advanced Settings (Content)', 'socialwire-article-generator' ); ?>
					</summary>
					<div class="socialwire-accordion-content">
						<form method="post" action="">
						<?php wp_nonce_field( 'socialwire_content_text_settings' ); ?>
						<table class="form-table">

							<tr>
								<th scope="row"><?php echo esc_html__( 'Heading Level Adjustment', 'socialwire-article-generator' ); ?></th>
								<td>
									<select name="heading_start_level" id="heading_start_level">
										<option value="h2" <?php selected( $heading_start_level ?? 'h2', 'h2' ); ?>><?php echo esc_html__( 'Start from h2 (Default)', 'socialwire-article-generator' ); ?></option>
										<option value="h3" <?php selected( $heading_start_level ?? 'h2', 'h3' ); ?>><?php echo esc_html__( 'Start from h3', 'socialwire-article-generator' ); ?></option>
										<option value="h4" <?php selected( $heading_start_level ?? 'h2', 'h4' ); ?>><?php echo esc_html__( 'Start from h4', 'socialwire-article-generator' ); ?></option>
										<option value="h5" <?php selected( $heading_start_level ?? 'h2', 'h5' ); ?>><?php echo esc_html__( 'Start from h5', 'socialwire-article-generator' ); ?></option>
										<option value="h6" <?php selected( $heading_start_level ?? 'h2', 'h6' ); ?>><?php echo esc_html__( 'Start from h6', 'socialwire-article-generator' ); ?></option>
									</select>
									<p class="description">
										<?php echo wp_kses_post( __( 'Adjusts heading levels in the content.<br><strong>System specification:</strong> Since titles use h1, content normally starts from h2.<br><strong>Example when "Start from h3" is selected:</strong><br>• h2 in content → converted to h3<br>• h3 in content → converted to h4<br>• h4 in content → converted to h5<br>• Similarly adjusted by one level<br><strong>Note:</strong> Please select according to the heading specifications of your target media.', 'socialwire-article-generator' ) ); ?>
									</p>
								</td>
							</tr>
						</table>
						<?php submit_button( __( 'Save Content Settings', 'socialwire-article-generator' ), 'secondary', 'submit_content_text_settings' ); ?>
					</form>
					</div>
				</details>
			</div>

			<!-- Custom Field Settings Section -->
			<div style="margin-top: 20px;">
				<details class="socialwire-accordion">
					<summary class="socialwire-accordion-header">
						<?php echo esc_html__( 'Custom Fields', 'socialwire-article-generator' ); ?>
					</summary>
					<div class="socialwire-accordion-content">
						<form method="post" action="">
						<?php wp_nonce_field( 'socialwire_custom_field_settings' ); ?>
						<table class="form-table">
							<tr>
								<th scope="row"><?php echo esc_html__( 'Featured Image', 'socialwire-article-generator' ); ?></th>
								<td>
									<select name="eyecatch_custom_field" id="eyecatch_custom_field">
										<option value=""><?php echo esc_html__( 'Do not save to custom field', 'socialwire-article-generator' ); ?></option>
										<?php
										$available_fields = $this->get_available_custom_fields();
										foreach ( $available_fields['image'] as $field_key => $field_label ) {
											echo '<option value="' . esc_attr( $field_key ) . '"' . selected( $eyecatch_custom_field, $field_key, false ) . '>' . esc_html( $field_label ) . '</option>';
										}
										?>
									</select>
									<p class="description">
										<?php echo wp_kses_post( __( 'Select the custom field to save the featured image.<br><strong>Note:</strong> The featured image will be saved to both the WordPress featured image and the selected custom field.', 'socialwire-article-generator' ) ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php echo esc_html__( 'Lead Text', 'socialwire-article-generator' ); ?></th>
								<td>
									<select name="lead_text_custom_field" id="lead_text_custom_field">
										<option value=""><?php echo esc_html__( 'Do not extract lead text', 'socialwire-article-generator' ); ?></option>
										<?php
										foreach ( $available_fields['text'] as $field_key => $field_label ) {
											echo '<option value="' . esc_attr( $field_key ) . '"' . selected( $lead_text_custom_field, $field_key, false ) . '>' . esc_html( $field_label ) . '</option>';
										}
										?>
									</select>
									<p class="description">
										<?php echo wp_kses_post( __( 'Select the custom field to save the lead text.<br><strong>Note:</strong> The text before the first heading will be extracted as lead text and saved to the selected custom field. This text will be removed from the main content.<br><strong>Behavior:</strong> If the content starts with a heading, no lead text will be extracted.', 'socialwire-article-generator' ) ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php echo esc_html__( 'Information Source', 'socialwire-article-generator' ); ?></th>
								<td>
									<select name="source_custom_field" id="source_custom_field">
										<option value=""><?php echo esc_html__( 'Do not save source information', 'socialwire-article-generator' ); ?></option>
										<?php
										foreach ( $available_fields['text'] as $field_key => $field_label ) {
											echo '<option value="' . esc_attr( $field_key ) . '"' . selected( $source_custom_field, $field_key, false ) . '>' . esc_html( $field_label ) . '</option>';
										}
										?>
									</select>
									<p class="description">
										<?php echo wp_kses_post( __( 'Select the custom field to save the information source URL.<br><strong>Note:</strong> The source URL from the RSS feed will be saved to the selected custom field.', 'socialwire-article-generator' ) ); ?>
									</p>
								</td>
							</tr>
						</table>
						<?php submit_button( __( 'Save Custom Field Settings', 'socialwire-article-generator' ), 'secondary', 'submit_custom_field' ); ?>
					</form>
					</div>
				</details>
			</div>

			<hr>

			<?php
			// Get manually edited posts.
			// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Acceptable for admin interface, limited by manual edits only.
			$manually_edited_posts = get_posts(
				array(
					'post_type'   => 'post',
					'meta_query'  => array(
						array(
							'key'     => 'pcn_manually_edited',
							'compare' => 'EXISTS',
						),
					),
					'numberposts' => -1,
					'post_status' => 'any',
				)
			);
			// phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query			// Get deleted document IDs.
			$deleted_document_ids = get_option( 'pcn_manually_deleted_document_ids', array() );
			$deleted_count        = count( $deleted_document_ids );

			// Count trash vs permanent delete.
			$trash_count            = 0;
			$permanent_delete_count = 0;
			foreach ( $deleted_document_ids as $document_id => $info ) {
				if ( is_array( $info ) && isset( $info['action'] ) ) {
					if ( 'wp_trash_post' === $info['action'] ) {
						++$trash_count;
					} else {
						++$permanent_delete_count;
					}
				} else {
					// Fallback: count as permanent delete if info is missing or malformed.
					++$permanent_delete_count;
				}
			}
			?>
			<div class="notice notice-info">
				<p><strong><?php echo esc_html__( 'Current Protection Status:', 'socialwire-article-generator' ); ?></strong></p>
				<ul>
					<?php /* translators: %s: number of articles */ ?>
					<li><?php printf( esc_html__( 'Articles protected by manual editing: %s items', 'socialwire-article-generator' ), '<strong>' . count( $manually_edited_posts ) . '</strong>' ); ?></li>
					<?php /* translators: %s: number of document IDs */ ?>
					<li><?php printf( esc_html__( 'Document IDs protected by manual deletion: %s items', 'socialwire-article-generator' ), '<strong>' . esc_html( $deleted_count ) . '</strong>' ); ?>
						<?php if ( $deleted_count > 0 ) : ?>
						<ul style="margin-left: 20px; margin-top: 5px;">
							<?php /* translators: %s: number of items moved to trash */ ?>
							<li><?php printf( esc_html__( 'Moved to trash: %s items', 'socialwire-article-generator' ), esc_html( $trash_count ) ); ?></li>
							<?php /* translators: %s: number of permanently deleted items */ ?>
							<li><?php printf( esc_html__( 'Permanently deleted: %s items', 'socialwire-article-generator' ), esc_html( $permanent_delete_count ) ); ?></li>
						</ul>
						<?php endif; ?>
					</li>
				</ul>
			</div>

			<!-- Plugin Memory Usage Information -->
			<div class="notice notice-info" style="margin-top: 15px;">
				<?php echo wp_kses_post( $this->get_memory_usage_display() ); ?>
			</div>


			<h2><?php echo esc_html__( 'Manual Execution', 'socialwire-article-generator' ); ?></h2>
			<p>
				<button type="button" id="manual-generate" class="button button-primary">
					<?php echo esc_html__( 'Generate Articles Now', 'socialwire-article-generator' ); ?>
				</button>
				<button type="button" id="manual-fetch" class="button button-secondary" style="margin-left: 10px;">
					<?php echo esc_html__( 'Fetch Articles Now', 'socialwire-article-generator' ); ?>
				</button>
				<span id="generate-status"></span>
			</p>

			<div id="generate-log" style="margin-top: 20px;"></div>

			<script type="text/javascript">
			jQuery(document).ready(function($) {
				'use strict';
				
				$('#manual-generate').on('click', function() {
					var $button = $(this);
					var $status = $('#generate-status');
					var $log = $('#generate-log');
					var $fetchButton = $('#manual-fetch');

					$button.prop('disabled', true);
					$fetchButton.prop('disabled', true);
					$status.html('<span style="color: blue;">' + <?php echo wp_json_encode( __( 'Article generation in progress...', 'socialwire-article-generator' ) ); ?> + '</span>');
					$log.html('<p>' + <?php echo wp_json_encode( __( 'Article generation process started...', 'socialwire-article-generator' ) ); ?> + '</p>');

					$.ajax({
						url: ajaxurl,
						type: 'POST',
						timeout: 300000,
						data: {
							action: 'socialwire_manual_generate',
							nonce: <?php echo wp_json_encode( wp_create_nonce( 'socialwire_generate_nonce' ) ); ?>
						},
						success: function(response) {
							$button.prop('disabled', false);
							$fetchButton.prop('disabled', false);
							if (response.success) {
								$status.html('<span style="color: green;">' + <?php echo wp_json_encode( __( 'Generation completed', 'socialwire-article-generator' ) ); ?> + '</span>');
								$log.html('<h3>' + <?php echo wp_json_encode( __( 'Execution Results:', 'socialwire-article-generator' ) ); ?> + '</h3><pre>' + response.data.log + '</pre>');
							} else {
								var errorMessage = response.data && response.data.message ? response.data.message : (response.data || <?php echo wp_json_encode( __( 'An error occurred', 'socialwire-article-generator' ) ); ?>);
								$status.html('<span style="color: red;">' + <?php echo wp_json_encode( __( 'Error: ', 'socialwire-article-generator' ) ); ?> + errorMessage + '</span>');
								if (response.data && response.data.log) {
									$log.html('<h3>' + <?php echo wp_json_encode( __( 'Error Details:', 'socialwire-article-generator' ) ); ?> + '</h3><pre>' + response.data.log + '</pre>');
								}
							}
						},
						error: function(xhr, status, error) {
							$button.prop('disabled', false);
							$fetchButton.prop('disabled', false);
							if (status === 'timeout') {
								$status.html('<span style="color: orange;">' + <?php echo wp_json_encode( __( 'Timeout occurred', 'socialwire-article-generator' ) ); ?> + '</span>');
							} else {
								$status.html('<span style="color: red;">' + <?php echo wp_json_encode( __( 'Communication error: ', 'socialwire-article-generator' ) ); ?> + error + '</span>');
							}
						}
					});
				});

				$('#manual-fetch').on('click', function() {
					var $button = $(this);
					var $status = $('#generate-status');
					var $log = $('#generate-log');
					var $generateButton = $('#manual-generate');

					$button.prop('disabled', true);
					$generateButton.prop('disabled', true);
					$status.html('<span style="color: blue;">' + <?php echo wp_json_encode( __( 'Article fetching in progress...', 'socialwire-article-generator' ) ); ?> + '</span>');
					$log.html('<p>' + <?php echo wp_json_encode( __( 'Article fetching process started...', 'socialwire-article-generator' ) ); ?> + '</p>');

					$.ajax({
						url: ajaxurl,
						type: 'POST',
						timeout: 300000,
						data: {
							action: 'socialwire_manual_fetch',
							nonce: <?php echo wp_json_encode( wp_create_nonce( 'socialwire_fetch_nonce' ) ); ?>
						},
						success: function(response) {
							$button.prop('disabled', false);
							$generateButton.prop('disabled', false);
							if (response.success) {
								$status.html('<span style="color: green;">' + <?php echo wp_json_encode( __( 'Fetch completed', 'socialwire-article-generator' ) ); ?> + '</span>');
								$log.html('<h3>' + <?php echo wp_json_encode( __( 'Execution Results:', 'socialwire-article-generator' ) ); ?> + '</h3><pre>' + response.data.log + '</pre>');
							} else {
								var errorMessage = response.data && response.data.message ? response.data.message : (response.data || <?php echo wp_json_encode( __( 'An error occurred', 'socialwire-article-generator' ) ); ?>);
								$status.html('<span style="color: red;">' + <?php echo wp_json_encode( __( 'Error: ', 'socialwire-article-generator' ) ); ?> + errorMessage + '</span>');
								if (response.data && response.data.log) {
									$log.html('<h3>' + <?php echo wp_json_encode( __( 'Error Details:', 'socialwire-article-generator' ) ); ?> + '</h3><pre>' + response.data.log + '</pre>');
								}
							}
						},
						error: function(xhr, status, error) {
							$button.prop('disabled', false);
							$generateButton.prop('disabled', false);
							if (status === 'timeout') {
								$status.html('<span style="color: orange;">' + <?php echo wp_json_encode( __( 'Timeout occurred', 'socialwire-article-generator' ) ); ?> + '</span>');
							} else {
								$status.html('<span style="color: red;">' + <?php echo wp_json_encode( __( 'Communication error: ', 'socialwire-article-generator' ) ); ?> + error + '</span>');
							}
						}
					});
				});
			});
			</script>
		</div>

		<?php
	}

	/**
	 * Execute manual article generation process with memory tracking.
	 */
	private function execute_manual_generation() {
		// Add memory checkpoint for manual execution.
		$this->add_memory_checkpoint('manual_execution_start');
		
		// Remove execution time limit for time-consuming operations.
		// phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Necessary for LLM API calls that require extended execution time.
		set_time_limit( 0 );
		wp_raise_memory_limit( 'admin' );
		
		$this->add_memory_checkpoint('after_memory_limit_raise');

		// Start debug log collection.
		$this->debug_logs = array();
		$this->socialwire_log( '=== Manual Execution Started ===' );
		
		// Simplified locale check.
		$this->socialwire_log( 'Current WordPress locale: ' . get_locale() );
		$this->socialwire_log( 'Text domain loaded: ' . ( is_textdomain_loaded( 'socialwire-article-generator' ) ? 'Yes' : 'No' ) );

		$log = array();

		// Start article generation process.
		$this->socialwire_log( 'Starting article generation process.' );
		$this->add_memory_checkpoint('before_generate_process');
		$generate_result = $this->execute_generate_process();
		$this->add_memory_checkpoint('after_generate_process');
		$log[]           = '=== Article Generation Process ===';
		$log[]           = $generate_result['message'];

		// Start RSS fetch process.
		$this->socialwire_log( 'Starting RSS fetch process.' );
		$this->add_memory_checkpoint('before_rss_fetch');
		$rss_result = $this->execute_rss_fetch_process();
		$this->add_memory_checkpoint('after_rss_fetch');
		$log[]      = "\n=== RSS Fetch Process ===";
		$log[]      = $rss_result['message'];

		// Start URL notification process.
		$this->socialwire_log( 'Starting URL notification process.' );
		$this->add_memory_checkpoint('before_url_notify');
		$url_result = $this->execute_url_notify_process();
		$this->add_memory_checkpoint('after_url_notify');
		$log[]      = "\n=== URL Notification Process ===";
		$log[]      = $url_result['message'];

		$this->add_memory_checkpoint('manual_execution_end');
		$this->socialwire_log( '=== Manual Execution Completed ===' );

		// Add memory usage information to logs.
		$memory_info = $this->get_memory_usage_info();
		$this->socialwire_log( sprintf(
			'Memory usage during execution: Plugin used %s, Peak %s',
			size_format($memory_info['plugin_usage']),
			size_format($memory_info['peak'])
		) );

		// Display results to user.
		echo '<div class="notice notice-info">';
		echo '<h3>' . esc_html__( 'Manual Execution Results', 'socialwire-article-generator' ) . '</h3>';
		echo '<pre>' . esc_html( implode( "\n", $log ) ) . '</pre>';
		
		if ( ! empty( $this->debug_logs ) ) {
			echo '<h4>' . esc_html__( 'Debug Log', 'socialwire-article-generator' ) . '</h4>';
			echo '<pre>' . esc_html( implode( "\n", $this->debug_logs ) ) . '</pre>';
		}
		
		// Display updated memory usage after execution.
		echo '<h4>' . esc_html__( 'Memory Usage After Execution', 'socialwire-article-generator' ) . '</h4>';
		echo wp_kses_post( $this->get_memory_usage_display() );
		
		echo '</div>';
	}

	/**
	 * Handle AJAX request for manual article generation. Memory optimized with tracking.
	 *
	 * @since 1.0.0
	 */
	public function ajax_manual_generate() {
		check_ajax_referer( 'socialwire_generate_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'socialwire-article-generator' ) );
		}

		// Add memory checkpoint for AJAX execution.
		$this->add_memory_checkpoint('ajax_generate_start');

		// Use WordPress standard memory management.
		wp_raise_memory_limit('admin');

		// Remove execution time limit for article generation API calls.
		// phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Necessary for article generation API calls that require extended execution time.
		set_time_limit( 0 );

		$this->add_memory_checkpoint('ajax_after_memory_setup');

		// Start debug log collection.
		$this->debug_logs = array();
		$this->socialwire_log( '=== Manual Article Generation Started ===' );

		$log = array();

		// Execute only article generation process.
		$this->socialwire_log( 'Starting article generation process.' );
		$this->add_memory_checkpoint('ajax_before_generate');
		$generate_result = $this->execute_generate_process();
		$this->add_memory_checkpoint('ajax_after_generate');
		$log[]           = '=== Article Generation Process ===';
		$log[]           = $generate_result['message'];

		// Add curl command for manual execution
		if ( ! empty( $generate_result['curl_command'] ) ) {
			$log[] = "\n=== cURL Command for Manual Execution ===";
			$log[] = $generate_result['curl_command'];
		}

		$this->socialwire_log( '=== Manual Article Generation Completed ===' );

		// Add memory usage to logs.
		$memory_info = $this->get_memory_usage_info();
		$memory_log = sprintf(
			"\n=== Memory Usage Report ===\nPlugin Memory Used: %s\nPeak Memory: %s\nCheckpoints: %d",
			size_format($memory_info['plugin_usage']),
			size_format($memory_info['peak']),
			$memory_info['checkpoints_count']
		);

		// Return result with debug logs including memory info.
		$detailed_log = $log;
		if ( ! empty( $this->debug_logs ) ) {
			$detailed_log[] = "\n=== Detailed Debug Log ===";
			$detailed_log   = array_merge( $detailed_log, $this->debug_logs );
		}
		$detailed_log[] = $memory_log;

		// Check the result to determine success/failure.
		if ( $generate_result['success'] ) {
			wp_send_json_success(
				array(
					'log' => implode( "\n", $detailed_log ),
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message' => $generate_result['message'],
					'log'     => implode( "\n", $detailed_log ),
				)
			);
		}
	}

	/**
	 * Handle AJAX request for manual RSS fetching and URL notification. Memory optimized.
	 *
	 * @since 1.0.0
	 */
	public function ajax_manual_fetch() {
		check_ajax_referer( 'socialwire_fetch_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'socialwire-article-generator' ) );
		}

		// Remove execution time limit for time-consuming operations.
		// phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Necessary for fetch operations that require extended execution time.
		set_time_limit( 0 );
		wp_raise_memory_limit( 'admin' );

		// Start debug log collection.
		$this->debug_logs = array();
		$this->socialwire_log( '=== Manual RSS Fetch Started ===' );

		$log = array();

		// Execute RSS fetch process.
		$this->socialwire_log( 'Starting RSS fetch process.' );
		$rss_result = $this->execute_rss_fetch_process();
		$log[]      = '=== RSS Fetch Process ===';
		$log[]      = $rss_result['message'];

		// Execute URL notification process.
		$this->socialwire_log( 'Starting URL notification process.' );
		$url_result = $this->execute_url_notify_process();
		$log[]      = "\n=== URL Notification Process ===";
		$log[]      = $url_result['message'];

		$this->socialwire_log( '=== Manual RSS Fetch Completed ===' );

		// Return result with debug logs.
		$detailed_log = $log;
		if ( ! empty( $this->debug_logs ) ) {
			$detailed_log[] = "\n=== Detailed Debug Log ===";
			$detailed_log   = array_merge( $detailed_log, $this->debug_logs );
		}

		wp_send_json_success(
			array(
				'log' => implode( "\n", $detailed_log ),
			)
		);
	}

	/**
	 * Scheduled article generation.
	 */
	public function scheduled_generate() {
		$this->socialwire_log( '=== SCHEDULED GENERATE STARTED ===' );

		$lock_id = $this->acquire_distributed_lock( 'article_generate', 900 );
		if ( ! $lock_id ) {
			$this->socialwire_log( 'Another instance is running article generation, skipping' );
			return;
		}

		try {
			$settings = get_option( $this->option_name, array() );

			if ( isset( $settings['auto_generate'] ) && $settings['auto_generate'] ) {
				// Check if we should execute based on mode
				$auto_exec_mode = isset( $settings['auto_exec_mode'] ) ? $settings['auto_exec_mode'] : 'interval';
				
				if ( 'hourly' === $auto_exec_mode ) {
					// Time specification mode - check if current hour is in the list
					$auto_exec_hours = isset( $settings['auto_exec_hours'] ) ? $settings['auto_exec_hours'] : array();
					// Use WordPress timezone setting for hour comparison
					$current_hour = intval( current_time( 'G' ) );
					
					if ( ! in_array( $current_hour, $auto_exec_hours, true ) ) {
						$this->socialwire_log( sprintf( 'Current hour (%d) is not in execution schedule, skipping', $current_hour ) );
						return;
					}
					
					$this->socialwire_log( sprintf( 'Current hour (%d) is in execution schedule, proceeding', $current_hour ) );
				}
				
				$result = $this->execute_generate_process();
				$this->socialwire_log( 'Generate process result: ' . $result['message'] );
			} else {
				$this->socialwire_log( 'Auto generation is disabled' );
			}
		} finally {
			$this->release_distributed_lock( 'article_generate', $lock_id );
			$this->socialwire_log( '=== SCHEDULED GENERATE FINISHED ===' );
		}
	}

	/**
	 * Scheduled RSS fetch.
	 */
	public function scheduled_rss_fetch() {
		$this->socialwire_log( '=== SCHEDULED RSS FETCH STARTED ===' );

		$lock_id = $this->acquire_distributed_lock( 'rss_fetch', 900 );
		if ( ! $lock_id ) {
			$this->socialwire_log( 'Another instance is running RSS fetch, skipping' );
			return;
		}

		try {
			$settings = get_option( $this->option_name, array() );

			if ( isset( $settings['auto_generate'] ) && $settings['auto_generate'] ) {
				// Check if we should execute based on mode
				$auto_exec_mode = isset( $settings['auto_exec_mode'] ) ? $settings['auto_exec_mode'] : 'interval';
				
				if ( 'hourly' === $auto_exec_mode ) {
					// Time specification mode - check if current hour is in the list
					$auto_exec_hours = isset( $settings['auto_exec_hours'] ) ? $settings['auto_exec_hours'] : array();
					// Use WordPress timezone setting for hour comparison
					$current_hour = intval( current_time( 'G' ) );
					
					if ( ! in_array( $current_hour, $auto_exec_hours, true ) ) {
						$this->socialwire_log( sprintf( 'Current hour (%d) is not in execution schedule, skipping RSS fetch', $current_hour ) );
						return;
					}
					
					$this->socialwire_log( sprintf( 'Current hour (%d) is in execution schedule, proceeding with RSS fetch', $current_hour ) );
				}
				
				$result = $this->execute_rss_fetch_process();
				$this->socialwire_log( 'RSS fetch result: ' . $result['message'] );
			} else {
				$this->socialwire_log( 'Auto generation is disabled' );
			}
		} finally {
			$this->release_distributed_lock( 'rss_fetch', $lock_id );
			$this->socialwire_log( '=== SCHEDULED RSS FETCH FINISHED ===' );
		}
	}

	/**
	 * Scheduled URL notification.
	 */
	public function scheduled_url_notify() {
		$this->socialwire_log( '=== SCHEDULED URL NOTIFY STARTED ===' );

		$lock_id = $this->acquire_distributed_lock( 'url_notify', 900 );
		if ( ! $lock_id ) {
			$this->socialwire_log( 'Another instance is running URL notify, skipping' );
			return;
		}

		try {
			$settings = get_option( $this->option_name, array() );

			if ( isset( $settings['auto_generate'] ) && $settings['auto_generate'] ) {
				// Check if we should execute based on mode
				$auto_exec_mode = isset( $settings['auto_exec_mode'] ) ? $settings['auto_exec_mode'] : 'interval';
				
				if ( 'hourly' === $auto_exec_mode ) {
					// Time specification mode - check if current hour is in the list
					$auto_exec_hours = isset( $settings['auto_exec_hours'] ) ? $settings['auto_exec_hours'] : array();
					// Use WordPress timezone setting for hour comparison
					$current_hour = intval( current_time( 'G' ) );
					
					if ( ! in_array( $current_hour, $auto_exec_hours, true ) ) {
						$this->socialwire_log( sprintf( 'Current hour (%d) is not in execution schedule, skipping URL notify', $current_hour ) );
						return;
					}
					
					$this->socialwire_log( sprintf( 'Current hour (%d) is in execution schedule, proceeding with URL notify', $current_hour ) );
				}
				
				$result = $this->execute_url_notify_process();
				$this->socialwire_log( 'URL notify result: ' . $result['message'] );
			} else {
				$this->socialwire_log( 'Auto generation is disabled' );
			}
		} finally {
			$this->release_distributed_lock( 'url_notify', $lock_id );
			$this->socialwire_log( '=== SCHEDULED URL NOTIFY FINISHED ===' );
		}
	}

	/**
	 * Convert h1 tags to h2 (required processing).
	 *
	 * @param string $content Content to process.
	 * @return string Processed content.
	 */
	private function convert_h1_to_h2( $content ) {
		if ( empty( $content ) ) {
			return $content;
		}

		// h1タグの数をカウント（変換前）.
		$h1_count = preg_match_all( '/<h1(\s[^>]*)?>/i', $content );

		// h1タグをh2に変換（開始タグと終了タグ両方）.
		$content = preg_replace( '/<h1(\s[^>]*)?>/i', '<h2$1>', $content );
		$content = preg_replace( '/<\/h1>/i', '</h2>', $content );

		// 変換結果をログ出力.
		if ( $h1_count > 0 ) {
			$this->socialwire_log( 'Converted ' . $h1_count . ' h1 tags to h2 tags in content' );
		}

		return $content;
	}

	/**
	 * Process content format (markdown detection and HTML conversion).
	 *
	 * @param string $content Content to process.
	 * @param string $title   Optional title for logging.
	 * @return string Processed content.
	 */
	private function process_content_format( $content, $title = '' ) {
		if ( empty( $content ) ) {
			return $content;
		}

		$content = $this->convert_h1_to_h2( $content );

		$markdown_patterns = array(
			'/^#{1,6}\s/',  // Header syntax (# ## ###).
			'/\n#{1,6}\s/', // Header syntax within line.
			'/\*{1,2}[^*]+\*{1,2}/', // Emphasis syntax (*text* **text**).
			'/\[[^\]]+\]\([^)]+\)/', // Link syntax markdown format [text](url).
			'/^[\s]*[-*+]\s/', // List syntax.
			'/\n[\s]*[-*+]\s/', // List syntax within line.
			'/^[\s]*\d+\.\s/', // Numbered list.
			'/\n[\s]*\d+\.\s/', // Numbered list within line.
		);

		$is_markdown = false;
		foreach ( $markdown_patterns as $pattern ) {
			if ( preg_match( $pattern, $content ) ) {
				$is_markdown = true;
				$this->socialwire_log( 'Markdown detected in content for: ' . $title . ' (pattern: ' . $pattern . ')' );
				break;
			}
		}

		if ( $is_markdown ) {
			$this->socialwire_log( 'Converting markdown to HTML for: ' . $title );
			return $this->convert_markdown_to_html( $content );
		}

		return $content;
	}

	/**
	 * Convert simple markdown to HTML.
	 *
	 * @param string $markdown Markdown content to convert.
	 * @return string HTML content.
	 */
	private function convert_markdown_to_html( $markdown ) {
		$html = $markdown;

		// Header conversion (# Header -> <h1>Header</h1>, ## Header -> <h2>Header</h2>, etc.).
		// Note: h1 conversion will be processed later in heading level adjustment.
		$html = preg_replace_callback(
			'/^(#{1,6})\s+(.+)$/m',
			function ( $matches ) {
				$level = strlen( $matches[1] );

				$this->socialwire_log( 'Converting markdown heading level ' . $level . ' for: ' . trim( $matches[2] ) );

				return '<h' . $level . '>' . trim( $matches[2] ) . '</h' . $level . '>';
			},
			$html
		);

		// Strong emphasis conversion (**text** -> <strong>text</strong>).
		$html = preg_replace( '/\*\*([^*]+)\*\*/', '<strong>$1</strong>', $html );

		// Italic conversion (*text* -> <em>text</em>).
		$html = preg_replace( '/\*([^*]+)\*/', '<em>$1</em>', $html );

		// Link conversion ([text](url) -> <a href="url">text</a>).
		$html = $this->convert_links_with_new_tab_support( $html );

		// List conversion.
		$html = preg_replace_callback(
			'/^([\s]*)([-*+])\s+(.+)$/m',
			function ( $matches ) {
				$indent  = $matches[1];
				$content = $matches[3];
				return $indent . '<li>' . $content . '</li>';
			},
			$html
		);

		// Numbered list conversion.
		$html = preg_replace_callback(
			'/^([\s]*)(\d+)\.\s+(.+)$/m',
			function ( $matches ) {
				$indent  = $matches[1];
				$content = $matches[3];
				return $indent . '<li>' . $content . '</li>';
			},
			$html
		);

		// Paragraph conversion.
		$paragraphs           = explode( "\n\n", $html );
		$processed_paragraphs = array();

		foreach ( $paragraphs as $paragraph ) {
			$paragraph = trim( $paragraph );
			if ( empty( $paragraph ) ) {
				continue;
			}

			// Check if already wrapped in HTML tags.
			if ( preg_match( '/^<(h[1-6]|ul|ol|li|img|div|p)/', $paragraph ) ) {
				$processed_paragraphs[] = $paragraph;
			} else {
				// Convert newlines to <br> and wrap in <p> tags.
				$paragraph              = str_replace( "\n", '<br>', $paragraph );
				$processed_paragraphs[] = '<p>' . $paragraph . '</p>';
			}
		}

		$html = implode( "\n", $processed_paragraphs );

		// List conversion.
		$html = preg_replace( '/(<li>.*?<\/li>(?:\s*<li>.*?<\/li>)*)/s', '<ul>$1</ul>', $html );

		$this->socialwire_log( 'Markdown conversion completed (h1->h2 conversion applied)' );
		return $html;
	}

	/**
	 * Execute article generation process.
	 */
	private function execute_generate_process() {
		$settings = get_option( $this->option_name, array() );

		if ( empty( $settings['genre_prompt'] ) || empty( $settings['style_prompt'] ) ) {
			return array(
				'success' => false,
				'message' => __( 'Prompts are not configured', 'socialwire-article-generator' ),
			);
		}

		$pcn_id = $this->get_environment_unique_id();
		if ( ! $pcn_id ) {
			return array(
				'success' => false,
				'message' => __( 'Failed to retrieve PCN ID for generation', 'socialwire-article-generator' ),
			);
		}

		$params = array(
			'pcn_id'          => $pcn_id,
			'genre_prompt'    => $settings['genre_prompt'],
			'style_prompt'    => $settings['style_prompt'],
			'output_language' => $settings['output_language'],
			'category_mode'   => $settings['category_mode'] ?? 'none',
			'plugin_version'  => $this->get_plugin_version(),
			'limit'           => 30, // Number of articles to generate.
		);

		if ( 'select' === $settings['category_mode'] && ! empty( $settings['selected_categories'] ) ) {
			$params['selected_categories'] = $this->build_category_data( $settings['selected_categories'] );
		} elseif ( ! empty( $settings['default_category'] ) ) {
			// If not category_mode, set the fixed category in the same format as selected_categories.
			$default_category = get_category( $settings['default_category'] );
			if ( $default_category ) {
				$params['selected_categories'] = array(
					array(
						'id'          => $default_category->term_id,
						'name'        => $default_category->name,
						'parent_name' => $default_category->parent ? get_category( $default_category->parent )->name : null,
						'hierarchy'   => $default_category->name,
					),
				);
			}
		}

		$api_url = 'https://pcn-api.cowriter.jp/api/plugins/request';

		$response = wp_remote_post(
			$api_url,
			array(
				'timeout' => 300,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body'    => wp_json_encode( $params ),
			)
		);

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => __( 'Article Generation API Error: ', 'socialwire-article-generator' ) . $response->get_error_message(),
			);
		}

		$body   = wp_remote_retrieve_body( $response );
		$result = json_decode( $body, true );

		if ( ! $result ) {
			return array(
				'success' => false,
				'message' => __( 'Failed to parse API response', 'socialwire-article-generator' ),
			);
		}

		// Send plugin user URL after successful generation request
		$this->send_plugin_user_url('generate');

		// Generate curl command for manual execution
		$curl_command = $this->generate_curl_command($api_url, $params);

		return array(
			'success'      => true,
			'message'      => __( 'Article generation request completed: OK', 'socialwire-article-generator' ),
			'curl_command' => $curl_command,
		);
	}

	/**
	 * Generate curl command string for manual API execution.
	 *
	 * @param string $api_url The API URL.
	 * @param array  $params  The request parameters.
	 * @return string The curl command.
	 * @since 1.1.10
	 */
	private function generate_curl_command( $api_url, $params ) {
		$json_body = wp_json_encode( $params, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT );
		// Escape single quotes for shell
		$escaped_json = str_replace( "'", "'\"'\"'", $json_body );

		$curl_command = sprintf(
			"curl -X POST '%s' \\\n  -H 'Content-Type: application/json' \\\n  -d '%s'",
			$api_url,
			$escaped_json
		);

		return $curl_command;
	}

	/**
	 * Execute RSS fetch process.
	 */
	private function execute_rss_fetch_process() {
		// Set RSS import start flag (used for manual edit detection).
		set_transient( 'socialwire_rss_import_in_progress', true, 3600 ); // 1 hour expiration

		$settings = get_option( $this->option_name, array() );

		$pcn_id = $this->get_environment_unique_id();
		if ( ! $pcn_id ) {
			return array(
				'success' => false,
				'message' => __( 'Failed to retrieve PCN ID', 'socialwire-article-generator' ),
			);
		}

		$api_url = 'https://pcn-api.cowriter.jp/api/plugins/rss?' . http_build_query(
			array(
				'pcn_id' => $pcn_id,
				'limit'  => 30,
			)
		);

		$response = wp_remote_get(
			$api_url,
			array(
				'timeout' => 60,
			)
		);

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => __( 'RSS API Error: ', 'socialwire-article-generator' ) . $response->get_error_message(),
			);
		}

		$body = wp_remote_retrieve_body( $response );

		// 手動実行時のみRSSの生XMLをログ出力（デバッグ用）.
		if ( isset( $this->debug_logs ) ) {
			$this->socialwire_log( '=== RSS Raw XML (first 1000 chars) ===' );
			$this->socialwire_log( substr( $body, 0, 1000 ) );
			$this->socialwire_log( '=== RSS Raw XML End ===' );
		}

		$xml = simplexml_load_string( $body );

		if ( false === $xml ) {
			return array(
				'success' => false,
				'message' => __( 'Failed to parse RSS', 'socialwire-article-generator' ),
			);
		}

		// Check RSS lastModified.
		$rss_last_modified = $this->get_rss_last_modified( $xml );
		if ( $rss_last_modified ) {
			$stored_last_modified = get_option( 'pcn_rss_last_modified' );

			if ( $stored_last_modified && $rss_last_modified <= $stored_last_modified ) {
				$this->socialwire_log( 'RSS not modified since last check. LastModified: ' . $rss_last_modified . ', Stored: ' . $stored_last_modified );
				delete_transient( 'socialwire_rss_import_in_progress' );
				return array(
					'success' => true,
					/* translators: %s: last modified date */
					'message' => sprintf( __( 'RSS not updated (LastModified: %s)', 'socialwire-article-generator' ), $rss_last_modified ),
				);
			}

			$this->socialwire_log( 'RSS has been modified. LastModified: ' . $rss_last_modified . ', Stored: ' . ( $stored_last_modified ? $stored_last_modified : 'none' ) );
		} else {
			// If lastModified is not found, fall back to pubDate-based comparison.
			$latest_pub_date = $this->get_latest_pub_date( $xml );
			if ( $latest_pub_date ) {
				$stored_latest_pub_date = get_option( 'pcn_rss_latest_pub_date' );

				if ( $stored_latest_pub_date && $latest_pub_date <= $stored_latest_pub_date ) {
					$this->socialwire_log( 'RSS not modified since last check (pubDate fallback). Latest pubDate: ' . $latest_pub_date . ', Stored: ' . $stored_latest_pub_date );
					delete_transient( 'socialwire_rss_import_in_progress' );
					return array(
						'success' => true,
						/* translators: %s: latest publication date */
						'message' => sprintf( __( 'RSS not updated (Latest pubDate: %s)', 'socialwire-article-generator' ), gmdate( 'Y-m-d H:i:s', $latest_pub_date ) ),
					);
				}

				$this->socialwire_log( 'RSS has new content (pubDate fallback). Latest pubDate: ' . $latest_pub_date . ', Stored: ' . ( $stored_latest_pub_date ? $stored_latest_pub_date : 'none' ) );
				$rss_last_modified = $latest_pub_date; // For later saving.
			} else {
				$this->socialwire_log( 'No lastModified or pubDate found in RSS, proceeding with processing' );
			}
		}

		$imported_count       = 0;
		$updated_count        = 0;
		$rss_latest_modified  = 0; // Track the latest lastModified from each item.
		$processed_doc_ids    = array(); // Track processed document_ids within this request to prevent duplicates.

		foreach ( $xml->channel->item as $item ) {
			$guid        = (string) $item->guid;
			$title       = (string) $item->title;
			$description = (string) $item->description;
			// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- pubDate is standard RSS property name
			$pub_date = (string) $item->pubDate;
			$content  = (string) $item->children( 'content', true )->encoded;

			// Detect and convert Markdown content.
			$content = $this->process_content_format( $content, $title );

			// Retrieve custom elements.
			$document_id          = '';
			$eyecatch_url         = '';
			$category_id_from_rss = 0;
			$item_last_modified   = '';
			$source_url           = '';

			foreach ( $item->children() as $child ) {
				switch ( $child->getName() ) {
					case 'documentId':
						$document_id = (string) $child;
						break;
					case 'eyecatchUrl':
						$eyecatch_url = (string) $child;
						break;
					case 'category':
						$category_id_from_rss = (int) $child;
						break;
					case 'lastModified':
						$item_last_modified = (string) $child;
						$timestamp          = strtotime( $item_last_modified );
						if ( $timestamp > $rss_latest_modified ) {
							$rss_latest_modified = $timestamp;
						}
						break;
					case 'source':
						$source_url = (string) $child;
						break;
				}
			}

			// Skip if document_id is empty
			if ( empty( $document_id ) ) {
				$this->socialwire_log( 'Empty document_id, skipping item: ' . $title );
				continue;
			}

			// Skip if already processed in this request (RSS may contain duplicates)
			if ( isset( $processed_doc_ids[ $document_id ] ) ) {
				$this->socialwire_log( 'Document already processed in this request, skipping duplicate: ' . $document_id . ' (Title: ' . $title . ')' );
				continue;
			}
			$processed_doc_ids[ $document_id ] = true;

			// Use transient-based lock to prevent duplicate creation from concurrent requests
			$doc_lock_key = 'socialwire_doc_' . md5( $document_id );
			$existing_lock = get_transient( $doc_lock_key );
			if ( $existing_lock ) {
				$this->socialwire_log( 'Document is being processed by another request: ' . $document_id . ' (Title: ' . $title . ')' );
				continue;
			}
			// Set lock for 5 minutes
			set_transient( $doc_lock_key, time(), 300 );

			try {
				// Debug: Log RSS parsing results.
				$this->socialwire_log(
					sprintf(
						'RSS Item - Title: %s, Document ID: %s, Category ID: %d, LastModified: %s',
						$title,
						$document_id,
						$category_id_from_rss,
						$item_last_modified
					)
				);

				// Check for existing posts (after acquiring lock to ensure consistency).
				$existing_post = $this->get_post_by_document_id( $document_id );

				if ( $existing_post ) {
					// 既存記事が見つかった場合はスキップ（更新しない）
					$this->socialwire_log( 'Post already exists, skipping: ' . $title . ' (ID: ' . $existing_post->ID . ', Status: ' . $existing_post->post_status . ')' );
					continue;
				}

				// 以降は新規記事作成の処理のみ
				// Check for manually deleted document_id.
				if ( $this->is_manually_deleted_document_id( $document_id ) ) {
					$this->socialwire_log( 'Skipping creation for manually deleted document_id: ' . $document_id . ' (Title: ' . $title . ')' );
					continue;
				}

				// Check daily article limit before creating new article
				if ( $this->is_daily_article_limit_reached() ) {
					$this->socialwire_log( 'Daily article limit reached. Skipping article creation: ' . $title );
					continue;
				}

				// Create new post - Start with draft for safety.
				// Step 1: Fix system bug (h1→h2 conversion).
				$processed_content = $this->convert_h1_to_h2( $content );

				// Step 2: Adjust heading levels based on user settings.
				$content_text_settings = get_option( 'socialwire_content_text_settings', array() );
				$heading_start_level   = isset( $content_text_settings['heading_start_level'] ) ? $content_text_settings['heading_start_level'] : 'h2';

				// Adjust heading levels (only for h2 and below).
				$adjusted_content = $this->adjust_heading_levels( $processed_content, $heading_start_level );

				// Step 3: Extract lead text if configured.
				$lead_text_result = $this->extract_lead_text( $adjusted_content );
				$lead_text        = $lead_text_result['lead_text'];
				$main_content     = $lead_text_result['main_content'];

				// Apply new tab setting to links.
				$final_content = $this->apply_new_tab_to_links( $main_content );

				$post_data = array(
					'post_title'   => $title,
					'post_content' => $final_content,
					'post_excerpt' => $description,
					'post_status'  => 'draft', // Always set to draft during processing.
					'post_date'    => gmdate( 'Y-m-d H:i:s', strtotime( $pub_date ) ),
					'post_author'  => $settings['post_author'] ?? get_current_user_id(),
				);

				$post_id = wp_insert_post( $post_data );
				if ( $post_id && ! is_wp_error( $post_id ) ) {
					// Process images within the content (heading level adjustment has already been applied).
					$image_processed_content = $this->download_content_images( $post_id, $final_content );
					if ( $image_processed_content !== $final_content ) {
						// If image URLs have changed, reapply link settings and update the post.
						$final_processed_content = $this->apply_new_tab_to_links( $image_processed_content );
						wp_update_post(
							array(
								'ID'           => $post_id,
								'post_content' => $final_processed_content,
							)
						);
					}

					$this->set_post_meta_and_category( $post_id, $document_id, $eyecatch_url, $category_id_from_rss, $settings, $lead_text, $source_url );

					// After all processing is complete, change the publication status according to settings.
					$this->finalize_post_status( $post_id, $settings );
					++$imported_count;
				}
			} finally {
				// Always release the document-level lock
				delete_transient( $doc_lock_key );
			}
		}

		// Clear RSS import completion flag.
		delete_transient( 'socialwire_rss_import_in_progress' );

		// Update RSS lastModified (considering values obtained at item level).
		if ( $rss_last_modified ) {
			update_option( 'pcn_rss_last_modified', $rss_last_modified );
			$this->socialwire_log( 'Updated stored RSS lastModified to: ' . $rss_last_modified );
		} elseif ( $rss_latest_modified > 0 ) {
			update_option( 'pcn_rss_last_modified', $rss_latest_modified );
			$this->socialwire_log( 'Updated stored RSS lastModified (from items) to: ' . $rss_latest_modified );
		}

		// Update RSS latest pubDate (considering values obtained at item level).
		$latest_pub_date = $this->get_latest_pub_date( $xml );
		if ( $latest_pub_date ) {
			update_option( 'pcn_rss_latest_pub_date', $latest_pub_date );
			$this->socialwire_log( 'Updated stored RSS latest pubDate to: ' . $latest_pub_date );
		}

		return array(
			'success' => true,
			/* translators: %1$d: number of new items, %2$d: number of updated items */
			'message' => sprintf( __( 'RSS processing completed: New %1$d items, Updated %2$d items', 'socialwire-article-generator' ), $imported_count, $updated_count ),
		);
	}

	/**
	 * Get the last modified timestamp from the RSS XML.
	 *
	 * @param SimpleXMLElement $xml RSS XML object.
	 * @return int|null Last modified timestamp or null if not found.
	 */
	private function get_rss_last_modified( $xml ) {
		try {
			$this->socialwire_log( 'Searching for lastModified in RSS XML structure...' );

			// 1. Look for lastModified at channel level.
			if ( isset( $xml->channel->lastModified ) ) {
				$last_modified = (string) $xml->channel->lastModified;
				$this->socialwire_log( 'Found RSS lastModified at channel level: ' . $last_modified );
				return strtotime( $last_modified );
			}

			// 2. Get the latest lastModified from each item.
			$latest_timestamp = 0;
			foreach ( $xml->channel->item as $item ) {
				// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- lastModified is standard RSS property name
				if ( isset( $item->lastModified ) ) {
					// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- lastModified is standard RSS property name
					$last_modified = (string) $item->lastModified;
					$timestamp     = strtotime( $last_modified );
					if ( $timestamp > $latest_timestamp ) {
						$latest_timestamp = $timestamp;
					}
					$this->socialwire_log( 'Found item lastModified: ' . $last_modified . ' (' . $timestamp . ')' );
				}
			}

			if ( $latest_timestamp > 0 ) {
				$this->socialwire_log( 'Latest item lastModified timestamp: ' . $latest_timestamp . ' (' . gmdate( 'Y-m-d H:i:s', $latest_timestamp ) . ')' );
				return $latest_timestamp;
			}

			// 3. Search including namespaces.
			$namespaces = $xml->getNamespaces( true );
			$this->socialwire_log( 'Available namespaces: ' . wp_json_encode( array_keys( $namespaces ) ) );

			foreach ( $namespaces as $prefix => $namespace ) {
				$channel_ns = $xml->channel->children( $namespace );
				// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- lastModified is standard RSS property name
				if ( isset( $channel_ns->lastModified ) ) {
					// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- lastModified is standard RSS property name
					$last_modified = (string) $channel_ns->lastModified;
					$this->socialwire_log( 'Found RSS lastModified in namespace ' . $prefix . ': ' . $last_modified );
					return strtotime( $last_modified );
				}
			}

			// 4. Search with XPath.
			$xpath_queries = array(
				'//lastModified',
				'//channel/lastModified',
				'//*[local-name()="lastModified"]',
			);

			foreach ( $xpath_queries as $query ) {
				$result = $xml->xpath( $query );
				if ( ! empty( $result ) ) {
					$last_modified = (string) $result[0];
					$this->socialwire_log( 'Found RSS lastModified via XPath (' . $query . '): ' . $last_modified );
					return strtotime( $last_modified );
				}
			}

			// 5. Dump and investigate all child elements of the channel.
			$this->socialwire_log( 'Debugging: Channel children elements:' );
			foreach ( $xml->channel->children() as $child ) {
				$this->socialwire_log( '  - ' . $child->getName() . ': ' . (string) $child );
			}

			// If not found, return null.
			$this->socialwire_log( 'No lastModified found in RSS after exhaustive search' );
			return null;
		} catch ( Exception $e ) {
			$this->socialwire_log( 'Error parsing RSS lastModified: ' . $e->getMessage() );
			return null;
		}
	}

	/**
	 * Retrieve the latest pubDate from RSS (for fallback).
	 *
	 * @param SimpleXMLElement $xml RSS XML object.
	 * @return int|null Latest pubDate timestamp or null if not found.
	 */
	private function get_latest_pub_date( $xml ) {
		try {
			$latest_timestamp = 0;

			foreach ( $xml->channel->item as $item ) {
				// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- pubDate is standard RSS property name
				$pub_date = (string) $item->pubDate;
				if ( ! empty( $pub_date ) ) {
					$timestamp = strtotime( $pub_date );
					if ( $timestamp > $latest_timestamp ) {
						$latest_timestamp = $timestamp;
					}
				}
			}

			if ( $latest_timestamp > 0 ) {
				$this->socialwire_log( 'Found latest pubDate in RSS: ' . gmdate( 'Y-m-d H:i:s', $latest_timestamp ) . ' (' . $latest_timestamp . ')' );
				return $latest_timestamp;
			}

			$this->socialwire_log( 'No valid pubDate found in RSS items' );
			return null;
		} catch ( Exception $e ) {
			$this->socialwire_log( 'Error parsing RSS pubDates: ' . $e->getMessage() );
			return null;
		}
	}

	/**
	 * Send plugin user URL to API
	 *
	 * @param string $trigger Trigger that called this function (activation, manual, scheduled)
	 * @return array Result array with success status and message
	 */
	private function send_plugin_user_url($trigger = 'unknown') {
		$pcn_id = $this->get_environment_unique_id();
		if (!$pcn_id) {
			return array(
				'success' => false,
				'message' => __('Failed to retrieve PCN ID for plugin user URL notification', 'socialwire-article-generator'),
			);
		}

		// Get the correct domain URL
		$plugin_user_url = $this->get_correct_domain_automatic();
		if (empty($plugin_user_url)) {
			// Fallback to WordPress settings if automatic detection fails
			$plugin_user_url = get_option('home');
			if (empty($plugin_user_url)) {
				$plugin_user_url = get_option('siteurl');
			}
		}

		if (empty($plugin_user_url)) {
			$this->socialwire_log('Could not determine plugin user URL for notification');
			return array(
				'success' => false,
				'message' => __('Could not determine site URL for notification', 'socialwire-article-generator'),
			);
		}

		$api_url = 'https://pcn-api.cowriter.jp/api/plugins/plugin-user-request';
		
		$params = array(
			'pcn_id' => $pcn_id,
			'plugin_user_url' => $plugin_user_url,
		);

		$this->socialwire_log('Sending plugin user URL to API - Trigger: ' . $trigger . ', URL: ' . $plugin_user_url);

		$response = wp_remote_post(
			$api_url,
			array(
				'timeout' => 30,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body' => wp_json_encode($params),
			)
		);

		if (is_wp_error($response)) {
			$error_message = 'Plugin user URL notification failed: ' . $response->get_error_message();
			$this->socialwire_log($error_message);
			return array(
				'success' => false,
				'message' => $error_message,
			);
		}

		$response_code = wp_remote_retrieve_response_code($response);
		$response_body = wp_remote_retrieve_body($response);

		if ($response_code >= 200 && $response_code < 300) {
			$this->socialwire_log('Successfully sent plugin user URL notification (HTTP ' . $response_code . ')');
			return array(
				'success' => true,
				'message' => 'Plugin user URL notification sent successfully',
			);
		} else {
			$decoded_body = json_decode($response_body, true);
			$error_detail = '';
			if ($decoded_body && isset($decoded_body['detail'])) {
				$error_detail = $decoded_body['detail'];
			} else {
				$error_detail = $response_body;
			}

			$error_message = 'Plugin user URL notification failed (HTTP ' . $response_code . '): ' . $error_detail;
			$this->socialwire_log($error_message);
			return array(
				'success' => false,
				'message' => $error_message,
			);
		}
	}

	/**
	 * Rewrite internal IP URLs to correct domain
	 *
	 * @param string $url Original URL
	 * @return string Corrected URL
	 */
	private function rewrite_internal_to_correct_url($url) {
		if (empty($url)) {
			return $url;
		}
		
		$parsed = wp_parse_url($url);
		if (!$parsed || !isset($parsed['host'])) {
			return $url;
		}
		
		$host = $parsed['host'];
		
		// 内部IP/ホストの判定
		if (!$this->is_internal_host($host)) {
			return $url;
		}
		
		$correct_domain = $this->get_correct_domain_automatic();
		if (empty($correct_domain)) {
			$this->socialwire_log('Could not determine correct domain automatically for URL: ' . $url);
			return $url; // 自動検出できない場合は元のURLを返す
		}
		
		// URLの再構築
		$new_url = $correct_domain;
		$new_url .= isset($parsed['path']) ? $parsed['path'] : '';
		$new_url .= isset($parsed['query']) ? '?' . $parsed['query'] : '';
		$new_url .= isset($parsed['fragment']) ? '#' . $parsed['fragment'] : '';
		
		$this->socialwire_log('Rewritten URL: ' . $url . ' -> ' . $new_url);
		
		return $new_url;
	}

	/**
	 * Check if host is internal IP or hostname
	 *
	 * @param string $host Host to check
	 * @return bool True if internal
	 */
	private function is_internal_host($host) {
		// IPアドレスの場合
		if (filter_var($host, FILTER_VALIDATE_IP)) {
			// プライベートIPの判定（内部ネットワーク）
			return !filter_var($host, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
		}
		
		// ホスト名の場合
		$internal_hostnames = array(
			'localhost',
			'host.docker.internal',
			'127.0.0.1',
			'::1'
		);
		
		return in_array($host, $internal_hostnames, true);
	}

	/**
	 * Cache correct domain when detected from browser requests
	 */
	private function cache_correct_domain_from_browser() {
		// ブラウザからのリクエスト時のみ実行（管理画面アクセス時）
		if (!is_admin() || wp_doing_cron() || wp_doing_ajax()) {
			return;
		}
		
		$current_url = $this->get_current_page_url();
		if (empty($current_url)) {
			return;
		}
		
		$parsed_url = wp_parse_url($current_url);
		if (!$parsed_url || !isset($parsed_url['scheme']) || !isset($parsed_url['host'])) {
			return;
		}
		
		$browser_domain = $parsed_url['scheme'] . '://' . $parsed_url['host'];
		
		// 内部IPでないことを確認
		if ($this->is_internal_url($browser_domain)) {
			$this->socialwire_log('Browser domain is internal, not caching: ' . $browser_domain);
			return;
		}
		
		// 既存の保存されたドメインと比較
		$cached_domain = get_option('socialwire_cached_browser_domain');
		if ($cached_domain !== $browser_domain) {
			update_option('socialwire_cached_browser_domain', $browser_domain);
			$this->socialwire_log('Cached correct domain from browser: ' . $browser_domain);
			
			// 新しいドメインが取得できた場合、すぐにプラグインユーザーURLを送信
			$this->send_plugin_user_url('browser_cache_update');
		}
	}

	/**
	 * Get current page URL from browser request
	 *
	 * @return string Current page URL or empty string
	 */
	private function get_current_page_url() {
		if (!isset($_SERVER['HTTP_HOST']) || !isset($_SERVER['REQUEST_URI'])) {
			return '';
		}
		
		$scheme = 'https'; // デフォルトはHTTPS
		
		// プロトコルの判定
		if (!empty($_SERVER['HTTP_X_FORWARDED_PROTO'])) {
			$scheme = sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_PROTO']));
		} elseif (!empty($_SERVER['HTTP_X_FORWARDED_SSL']) && sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_SSL'])) === 'on') {
			$scheme = 'https';
		} elseif (!empty($_SERVER['HTTPS']) && sanitize_text_field(wp_unslash($_SERVER['HTTPS'])) !== 'off') {
			$scheme = 'https';
		} elseif (!empty($_SERVER['SERVER_PORT']) && intval($_SERVER['SERVER_PORT']) === 80) {
			$scheme = 'http';
		}
		
		$host = sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST']));
		
		return $scheme . '://' . $host;
	}

	/**
	 * Get correct domain URL (automatic detection with browser cache priority)
	 *
	 * @return string Correct domain or empty string
	 */
	private function get_correct_domain_automatic() {
		
		// 1. WordPress定数から取得
		if (defined('WP_HOME') && !$this->is_internal_url(WP_HOME)) {
			$domain = rtrim(WP_HOME, '/');
			$this->socialwire_log('Using domain from WP_HOME: ' . $domain);
			return $domain;
		}
		
		if (defined('WP_SITEURL') && !$this->is_internal_url(WP_SITEURL)) {
			$domain = rtrim(WP_SITEURL, '/');
			$this->socialwire_log('Using domain from WP_SITEURL: ' . $domain);
			return $domain;
		}
		
		// 2. WordPress設定（内部IPでない場合のみ）
		$home_url = get_option('home');
		if (!$this->is_internal_url($home_url)) {
			$domain = rtrim($home_url, '/');
			$this->socialwire_log('Using domain from home option: ' . $domain);
			return $domain;
		}
		
		$site_url = get_option('siteurl');
		if (!$this->is_internal_url($site_url)) {
			$domain = rtrim($site_url, '/');
			$this->socialwire_log('Using domain from siteurl option: ' . $domain);
			return $domain;
		}
		
		// 0. ブラウザから取得したドメインを最優先（新規追加）
		$browser_domain = get_option('socialwire_cached_browser_domain');
		if (!empty($browser_domain) && !$this->is_internal_url($browser_domain)) {
			$domain = rtrim($browser_domain, '/');
			$this->socialwire_log('Using cached browser domain: ' . $domain);
			return $domain;
		}
		
		// 3. 環境変数（Docker等）
		$env_domain = getenv('WORDPRESS_DOMAIN');
		if (!empty($env_domain) && !$this->is_internal_url($env_domain)) {
			$domain = rtrim($env_domain, '/');
			$this->socialwire_log('Using domain from environment variable: ' . $domain);
			return $domain;
		}
		
		// 4. サーバー環境から推測（最後の手段）
		$guessed_domain = $this->guess_correct_domain();
		if (!empty($guessed_domain) && !$this->is_internal_url($guessed_domain)) {
			$this->socialwire_log('Using guessed domain from server environment: ' . $guessed_domain);
			return $guessed_domain;
		}
		
		
		$this->socialwire_log('Could not determine correct domain from any source');
		return '';
	}

	/**
	 * Check if URL contains internal IP or localhost
	 *
	 * @param string $url URL to check
	 * @return bool True if URL is internal
	 */
	private function is_internal_url($url) {
		if (empty($url)) {
			return true;
		}
		
		$parsed = wp_parse_url($url);
		if (!$parsed || !isset($parsed['host'])) {
			return true;
		}
		
		return $this->is_internal_host($parsed['host']);
	}

	/**
	 * Guess correct domain from server environment
	 *
	 * @return string Guessed domain URL
	 */
	private function guess_correct_domain() {
		$scheme = 'https'; // デフォルトはHTTPS
		$host = '';
		
		// リバースプロキシ環境での正しいホスト名取得（優先順位順）
		$host_headers = array(
			'HTTP_X_FORWARDED_HOST',    // 最も一般的
			'HTTP_X_ORIGINAL_HOST',     // Cloudflare等
			'HTTP_X_REAL_IP',          // Nginx
			'HTTP_X_FORWARDED_FOR',    // 複数プロキシ経由
			'HTTP_HOST',               // 標準
			'SERVER_NAME'              // フォールバック
		);
		
		foreach ($host_headers as $header) {
			if (!empty($_SERVER[$header])) {
				$host = sanitize_text_field(wp_unslash($_SERVER[$header]));
				// X-Forwarded-Forの場合は最初のIPを使用
				if ($header === 'HTTP_X_FORWARDED_FOR') {
					$host = trim(explode(',', $host)[0]);
				}
				// 取得したホストが内部IPでないかチェック
				if (!$this->is_internal_host($host)) {
					break;
				} else {
					$host = ''; // 内部IPの場合は次のヘッダーを試す
				}
			}
		}
		
		// ホストが取得できない、または内部IPの場合は諦める
		if (empty($host) || $this->is_internal_host($host)) {
			$this->socialwire_log('Could not determine external host from server environment');
			return '';
		}
		
		// プロトコルの判定
		if (!empty($_SERVER['HTTP_X_FORWARDED_PROTO'])) {
			$scheme = sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_PROTO']));
		} elseif (!empty($_SERVER['HTTP_X_FORWARDED_SSL']) && sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_SSL'])) === 'on') {
			$scheme = 'https';
		} elseif (!empty($_SERVER['HTTPS']) && sanitize_text_field(wp_unslash($_SERVER['HTTPS'])) !== 'off') {
			$scheme = 'https';
		} else {
			// 現代では基本的にHTTPS
			$scheme = 'https';
		}
		
		// ポート番号を除去（標準ポート以外の場合は残す）
		$host = preg_replace('/:(80|443)$/', '', $host);
		
		$guessed_url = $scheme . '://' . $host;
		$this->socialwire_log('Guessed domain from server environment: ' . $guessed_url);
		
		return $guessed_url;
	}

	/**
	 * Execute URL notification process.
	 */
	private function execute_url_notify_process() {
		$settings = get_option( $this->option_name, array() );

		$pcn_id = $this->get_environment_unique_id();
		if ( ! $pcn_id ) {
			return array(
				'success' => false,
				'message' => __( 'Failed to retrieve PCN ID for notification', 'socialwire-article-generator' ),
			);
		}

		// Retrieve all PCN articles (to check notification status).
		// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Acceptable for cron operation, needed for notification status.
		$all_posts = get_posts(
			array(
				'post_type'   => 'post',
				'post_status' => 'publish',
				'meta_query'  => array(
					array(
						'key'     => 'pcn_document_id',
						'compare' => 'EXISTS',
					),
				),
				'numberposts' => -1,  // Retrieve all posts.
			)
		);
		// phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		// Retrieve posts to notify.
		// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Acceptable for cron operation, limited by numberposts.
		$posts = get_posts(
			array(
				'post_type'   => 'post',
				'post_status' => 'publish',
				'meta_query'  => array(
					array(
						'key'     => 'pcn_document_id',
						'compare' => 'EXISTS',
					),
					array(
						'relation' => 'OR',
						array(
							'key'     => 'pcn_url_notified',
							'compare' => 'NOT EXISTS',
						),
						array(
							'key'     => 'pcn_url_notified',
							'value'   => '1',
							'compare' => '!=',
						),
					),
				),
				'numberposts' => 50,
			)
		);
		// phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		$notified_count         = 0;
		$failed_count           = 0;
		$skipped_count          = 0;
		$already_notified_count = count( $all_posts ) - count( $posts );
		$api_url                = 'https://pcn-api.cowriter.jp/api/plugins/notify';

		$this->socialwire_log( 'Total PCN posts found: ' . count( $all_posts ) );
		$this->socialwire_log( 'Already notified posts: ' . $already_notified_count );
		$this->socialwire_log( 'Posts to notify: ' . count( $posts ) );

		foreach ( $posts as $post ) {
			$document_id = get_post_meta( $post->ID, 'pcn_document_id', true );
			if ( empty( $document_id ) ) {
				$this->socialwire_log( 'Post ID ' . $post->ID . ' has no document_id, skipping' );
				++$skipped_count;
				continue;
			}

			$published_url = get_permalink( $post->ID );
			
			// 内部IPを自動修正
			$published_url = $this->rewrite_internal_to_correct_url($published_url);

			$params = array(
				'published_url' => $published_url,
				'pcn_id'        => $pcn_id,
			);

			$this->socialwire_log( 'Notifying URL for post ID ' . $post->ID . ' (document_id: ' . $document_id . ') - URL: ' . $published_url );

			$response = wp_remote_post(
				$api_url,
				array(
					'timeout' => 30,
					'headers' => array(
						'Content-Type' => 'application/json',
					),
					'body'    => wp_json_encode( $params ),
				)
			);

			if ( is_wp_error( $response ) ) {
				$this->socialwire_log( 'URL notification failed for post ID ' . $post->ID . ': ' . $response->get_error_message() );
				++$failed_count;
			} else {
				$response_code = wp_remote_retrieve_response_code( $response );
				$response_body = wp_remote_retrieve_body( $response );

				if ( $response_code >= 200 && $response_code < 300 ) {
					update_post_meta( $post->ID, 'pcn_url_notified', '1' );
					$this->socialwire_log( 'Successfully notified URL for post ID ' . $post->ID . ' (HTTP ' . $response_code . ')' );
					++$notified_count;
				} else {
					// Decode the response body and display detailed information.
					$decoded_body = json_decode( $response_body, true );
					$error_detail = '';
					if ( $decoded_body && isset( $decoded_body['detail'] ) ) {
						$error_detail = $decoded_body['detail'];
					} else {
						$error_detail = $response_body;
					}

					$this->socialwire_log( 'URL notification failed for post ID ' . $post->ID . ' (HTTP ' . $response_code . ')' );
					$this->socialwire_log( 'Document ID: ' . $document_id );
					$this->socialwire_log( 'Published URL: ' . $published_url );
					$this->socialwire_log( 'Error detail: ' . $error_detail );
					++$failed_count;
				}
			}
		}

		return array(
			'success' => true,
			'message' => sprintf(
				/* translators: %1$d: new notifications, %2$d: failed items, %3$d: skipped items, %4$d: already notified items, %5$d: total items */
				__( 'URL notification processing completed: New notifications %1$d items, Failed %2$d items, Skipped %3$d items, Already notified %4$d items (Total %5$d items)', 'socialwire-article-generator' ),
				$notified_count,
				$failed_count,
				$skipped_count,
				$already_notified_count,
				count( $all_posts )
			),
		);
	}

	/**
	 * Retrieve article by document_id.
	 *
	 * @param string $document_id Document ID to search for.
	 * @return WP_Post|null Found post object or null if not found.
	 */
	private function get_post_by_document_id( $document_id ) {
		if ( empty( $document_id ) ) {
			return null;
		}

		// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Acceptable, limited by numberposts=1 and specific meta key.
		$posts = get_posts(
			array(
				'meta_query'  => array(
					array(
						'key'   => 'pcn_document_id',
						'value' => $document_id,
					),
				),
				'post_type'   => 'post',
				'post_status' => 'any', // Check all statuses including trash, pending, future, etc.
				'numberposts' => 1,
			)
		);
		// phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		return ! empty( $posts ) ? $posts[0] : null;
	}

	/**
	 * Set article metadata and categories.
	 *
	 * @param int    $post_id              Post ID.
	 * @param string $document_id          Document ID.
	 * @param string $eyecatch_url         Eyecatch image URL.
	 * @param int    $category_id_from_rss Category ID from RSS.
	 * @param array  $settings             Plugin settings array.
	 * @param string $lead_text            Lead text extracted from content.
	 * @param string $source_url           Source URL from RSS.
	 */
	private function set_post_meta_and_category( $post_id, $document_id, $eyecatch_url, $category_id_from_rss, $settings, $lead_text = '', $source_url = '' ) {
		// メタデータ設定.
		update_post_meta( $post_id, 'pcn_document_id', $document_id );
		update_post_meta( $post_id, 'pcn_generated', '1' );
		if ( ! empty( $eyecatch_url ) ) {
			update_post_meta( $post_id, 'pcn_eyecatch_url', $eyecatch_url );
		}

		// Get custom field settings
		$custom_field_settings = get_option( 'socialwire_custom_field_settings', array() );
		$eyecatch_custom_field = isset( $custom_field_settings['eyecatch_custom_field'] ) ? $custom_field_settings['eyecatch_custom_field'] : '';
		$lead_text_custom_field = isset( $custom_field_settings['lead_text_custom_field'] ) ? $custom_field_settings['lead_text_custom_field'] : '';
		$source_custom_field = isset( $custom_field_settings['source_custom_field'] ) ? $custom_field_settings['source_custom_field'] : '';

		// Save lead text to custom field if configured and lead text exists
		if ( ! empty( $lead_text_custom_field ) && ! empty( $lead_text ) ) {
			update_post_meta( $post_id, $lead_text_custom_field, $lead_text );
			$this->socialwire_log( sprintf( 'Post ID %d - Saved lead text to custom field "%s": %s', $post_id, $lead_text_custom_field, substr( $lead_text, 0, 100 ) . '...' ) );
		}

		// Save source URL to custom field if configured and source URL exists
		if ( ! empty( $source_custom_field ) && ! empty( $source_url ) ) {
			update_post_meta( $post_id, $source_custom_field, $source_url );
			$this->socialwire_log( sprintf( 'Post ID %d - Saved source URL to custom field "%s": %s', $post_id, $source_custom_field, $source_url ) );
		}

		// Debug: Log category information.
		$this->socialwire_log(
			sprintf(
				'Post ID %d - Category from RSS: %d, Category mode: %s, Default category: %d',
				$post_id,
				$category_id_from_rss,
				$settings['category_mode'] ?? 'not_set',
				$settings['default_category'] ?? 0
			)
		);

		// Category settings.
		$categories = array();

		if ( 'none' === $settings['category_mode'] ) {
			// Use fixed category.
			if ( ! empty( $settings['default_category'] ) ) {
				$categories[] = $settings['default_category'];
			}
		} else {
			// Prioritize categories from RSS (when category_mode is 'select').
			if ( ! empty( $category_id_from_rss ) ) {
				// Check if the category ID obtained from RSS actually exists.
				if ( get_category( $category_id_from_rss ) ) {
					$categories[] = $category_id_from_rss;
				}
			}

			// Use fixed category if no category is found.
			if ( empty( $categories ) && ! empty( $settings['default_category'] ) ) {
				$categories[] = $settings['default_category'];
			}
		}

		if ( ! empty( $categories ) ) {
			wp_set_post_categories( $post_id, $categories );
			$this->socialwire_log( sprintf( 'Post ID %d - Set categories: %s', $post_id, implode( ',', $categories ) ) );
		} else {
			$this->socialwire_log( sprintf( 'Post ID %d - No categories to set', $post_id ) );
		}

		// Set featured image and save to custom field if configured.
		if ( ! empty( $eyecatch_url ) && ! get_post_thumbnail_id( $post_id ) ) {
			$attachment_id = $this->download_and_set_featured_image( $post_id, $eyecatch_url );
			
			// Save featured image to custom field if configured and attachment was created
			if ( ! empty( $eyecatch_custom_field ) && $attachment_id ) {
				update_post_meta( $post_id, $eyecatch_custom_field, $attachment_id );
				$this->socialwire_log( sprintf( 'Post ID %d - Saved featured image to custom field "%s": %d', $post_id, $eyecatch_custom_field, $attachment_id ) );
			}
		}
	}

	/**
	 * Set post status after article processing is complete.
	 *
	 * @param int   $post_id  Post ID.
	 * @param array $settings Plugin settings array.
	 */
	private function finalize_post_status( $post_id, $settings ) {
		$desired_status = $settings['post_status'] ?? 'draft';

		$this->socialwire_log( sprintf( 'Post ID %d - Finalizing status to: %s', $post_id, $desired_status ) );

		if ( 'publish' === $desired_status ) {
			// Publish only if specified in settings.
			$result = wp_update_post(
				array(
					'ID'          => $post_id,
					'post_status' => 'publish',
				)
			);

			if ( $result && ! is_wp_error( $result ) ) {
				$this->socialwire_log( sprintf( 'Post ID %d - Successfully published', $post_id ) );
			} else {
				$this->socialwire_log( sprintf( 'Post ID %d - Failed to publish', $post_id ) );
			}
		} else {
			// Keep as draft.
			$this->socialwire_log( sprintf( 'Post ID %d - Keeping as draft', $post_id ) );
		}
	}


	/**
	 * Download and set featured image.
	 *
	 * @param int    $post_id   Post ID.
	 * @param string $image_url Image URL.
	 * @return int|false Attachment ID on success, false on failure.
	 */
	private function download_and_set_featured_image( $post_id, $image_url ) {
		if ( empty( $image_url ) ) {
			return false;
		}

		// Check for existing image.
		$existing_attachment = $this->get_attachment_by_url( $image_url );
		if ( $existing_attachment ) {
			$this->socialwire_log( 'Featured image already exists, using existing attachment: ' . $image_url . ' (ID: ' . $existing_attachment->ID . ')' );

			// Apply resize process even for existing images (according to featured image settings).
			$resize_result = $this->resize_image_if_needed( $existing_attachment->ID, 'featured' );
			$this->socialwire_log( 'Existing featured image resized URL: ' . ( $resize_result['url'] ? $resize_result['url'] : 'none' ) );

			$result = set_post_thumbnail( $post_id, $resize_result['attachment_id'] );
			if ( $result ) {
				$this->socialwire_log( 'Successfully set existing featured image for post ID: ' . $post_id );
				return $resize_result['attachment_id'];
			}
			return false;
		}

		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		$attachment_id = media_sideload_image( $image_url, $post_id, null, 'id' );

		if ( is_wp_error( $attachment_id ) ) {
			$this->socialwire_log( 'Featured image download failed: ' . $image_url . ' - ' . $attachment_id->get_error_message() );
			return false;
		}

		// Save original URL to meta field (to prevent duplicates).
		update_post_meta( $attachment_id, 'pcn_original_image_url', $image_url );

		// Apply resize process (according to featured image settings).
		$resize_result = $this->resize_image_if_needed( $attachment_id, 'featured' );
		$this->socialwire_log( 'Featured image resized URL: ' . ( $resize_result['url'] ? $resize_result['url'] : 'none' ) );

		$result = set_post_thumbnail( $post_id, $resize_result['attachment_id'] );
		if ( $result ) {
			$this->socialwire_log( 'Successfully set featured image for post ID: ' . $post_id );
			return $resize_result['attachment_id'];
		}

		return false;
	}

	/**
	 * Image resize process (with format conversion support).
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $image_type    Image type (featured or content).
	 */
	private function resize_image_if_needed( $attachment_id, $image_type = 'content' ) {
		// Get settings based on image type.
		if ( 'featured' === $image_type ) {
			$settings = get_option( 'socialwire_featured_image_settings', array() );
		} else {
			$settings = get_option( 'socialwire_content_image_settings', array() );
		}

		$max_width     = isset( $settings['max_image_width'] ) ? intval( $settings['max_image_width'] ) : 1200;
		$max_height    = isset( $settings['max_image_height'] ) ? intval( $settings['max_image_height'] ) : 800;
		$quality       = isset( $settings['resize_quality'] ) ? intval( $settings['resize_quality'] ) : 90;
		$target_format = isset( $settings['resize_format'] ) ? $settings['resize_format'] : 'original';

		// Check for format conversion settings.
		$needs_format_conversion = ( 'original' !== $target_format );

		// If resizing is disabled and format conversion is not needed, return original info.
		if ( ( ! isset( $settings['enable_image_resize'] ) || ! $settings['enable_image_resize'] ) && ! $needs_format_conversion ) {
			return array(
				'url'           => $this->get_correct_attachment_url( $attachment_id ),
				'attachment_id' => $attachment_id,
			);
		}

		$this->socialwire_log( 'Checking if image needs resize: ID=' . $attachment_id . ', max_width=' . $max_width . ', max_height=' . $max_height . ', format=' . $target_format );

		$file_path = get_attached_file( $attachment_id );
		if ( ! $file_path || ! file_exists( $file_path ) ) {
			$this->socialwire_log( 'File not found for attachment ID: ' . $attachment_id );
			return array(
				'url'           => $this->get_correct_attachment_url( $attachment_id ),
				'attachment_id' => $attachment_id,
			);
		}

		// Get image info.
		$image_info = getimagesize( $file_path );
		if ( ! $image_info ) {
			$this->socialwire_log( 'Could not get image info for: ' . $file_path );
			return array(
				'url'           => $this->get_correct_attachment_url( $attachment_id ),
				'attachment_id' => $attachment_id,
			);
		}

		$current_width  = $image_info[0];
		$current_height = $image_info[1];
		$mime_type      = $image_info['mime'];

		$this->socialwire_log( 'Original image size: ' . $current_width . 'x' . $current_height . ' (' . $mime_type . ')' );

		// Check if resizing is needed (only if resize feature is enabled).
		$resize_enabled = isset( $settings['enable_image_resize'] ) && $settings['enable_image_resize'];
		$needs_resize   = $resize_enabled && ( $current_width > $max_width || $current_height > $max_height );

		if ( ! $needs_resize && ! $needs_format_conversion ) {
			$this->socialwire_log( 'Image is within size limits and no format conversion needed' );
			return array(
				'url'           => $this->get_correct_attachment_url( $attachment_id ),
				'attachment_id' => $attachment_id,
			);
		}

		// Check if resized/converted version already exists.
		$resized_meta_key            = 'pcn_resized_' . $max_width . 'x' . $max_height . '_' . $quality . '_' . $target_format;
		$resized_attachment_meta_key = $resized_meta_key . '_attachment_id';

		$existing_resized_url           = get_post_meta( $attachment_id, $resized_meta_key, true );
		$existing_resized_attachment_id = get_post_meta( $attachment_id, $resized_attachment_meta_key, true );

		if ( $existing_resized_url && $existing_resized_attachment_id ) {
			// Check if existing attachment exists.
			if ( get_post( $existing_resized_attachment_id ) ) {
				$this->socialwire_log( 'Resized/converted version already exists: ' . $existing_resized_url . ' (Attachment ID: ' . $existing_resized_attachment_id . ')' );
				return array(
					'url'           => $existing_resized_url,
					'attachment_id' => $existing_resized_attachment_id,
				);
			} else {
				// If the attachment has been deleted, clear the metadata.
				delete_post_meta( $attachment_id, $resized_meta_key );
				delete_post_meta( $attachment_id, $resized_attachment_meta_key );
				$this->socialwire_log( 'Existing resized attachment was deleted, will create new one' );
			}
		}

		// Processing starts.
		$this->socialwire_log( 'Starting image resize/conversion process' );

		// Use WordPress image editor.
		$image_editor = wp_get_image_editor( $file_path );
		if ( is_wp_error( $image_editor ) ) {
			$this->socialwire_log( 'Failed to get image editor: ' . $image_editor->get_error_message() );
			return array(
				'url'           => $this->get_correct_attachment_url( $attachment_id ),
				'attachment_id' => $attachment_id,
			);
		}

		// Process animated GIFs.
		if ( 'image/gif' === $mime_type && $this->is_animated_gif( $file_path ) ) {
			$this->socialwire_log( 'Detected animated GIF, will use first frame only' );
			// WordPress image editor automatically uses the first frame.
		}

		// If resizing is needed, proceed.
		$new_width  = $current_width;
		$new_height = $current_height;

		if ( $needs_resize ) {
			// Maintain aspect ratio while calculating new size.
			$ratio_width  = $max_width / $current_width;
			$ratio_height = $max_height / $current_height;
			$ratio        = min( $ratio_width, $ratio_height );

			$new_width  = intval( $current_width * $ratio );
			$new_height = intval( $current_height * $ratio );

			$this->socialwire_log( 'Calculated new size: ' . $new_width . 'x' . $new_height . ' (ratio: ' . $ratio . ')' );

			$resize_result = $image_editor->resize( $new_width, $new_height, false );
			if ( is_wp_error( $resize_result ) ) {
				$this->socialwire_log( 'Failed to resize image: ' . $resize_result->get_error_message() );
				return array(
					'url'           => $this->get_correct_attachment_url( $attachment_id ),
					'attachment_id' => $attachment_id,
				);
			}
		}

		// Determine output format and extension.
		$output_format    = $this->determine_output_format( $target_format, $mime_type );
		$output_extension = $this->get_extension_for_format( $output_format );

		$this->socialwire_log( 'Output format: ' . $output_format . ', extension: ' . $output_extension );

		// If the output format is not capable of transparent backgrounds, convert transparent images to white.
		if ( $this->needs_background_conversion( $output_format, $mime_type ) ) {
			$this->socialwire_log( 'Converting transparent background to white' );
			$this->convert_transparent_to_white( $image_editor );
		}

		// Set quality for JPEG and WebP formats.
		if ( in_array( $output_format, array( 'image/jpeg', 'image/webp' ), true ) ) {
			$image_editor->set_quality( $quality );
		}

		// Generate file name.
		$upload_dir        = wp_upload_dir();
		$path_info         = pathinfo( $file_path );
		$suffix            = '_resized_' . ( $needs_resize ? ( $new_width . 'x' . $new_height ) : 'converted' );
		$resized_filename  = $path_info['filename'] . $suffix . '.' . $output_extension;
		$resized_file_path = $path_info['dirname'] . DIRECTORY_SEPARATOR . $resized_filename;

		// Set save options.
		$save_options = array();
		if ( 'image/jpeg' === $output_format ) {
			$save_options['quality'] = $quality;
		} elseif ( 'image/webp' === $output_format ) {
			$save_options['quality'] = $quality;
		}

		// Save resized/converted image.
		$save_result = $image_editor->save( $resized_file_path, $output_format );
		if ( is_wp_error( $save_result ) ) {
			$this->socialwire_log( 'Failed to save resized/converted image: ' . $save_result->get_error_message() );
			return array(
				'url'           => $this->get_correct_attachment_url( $attachment_id ),
				'attachment_id' => $attachment_id,
			);
		}

		// Generate URL for resized image.
		$resized_url = str_replace( $upload_dir['basedir'], '', $resized_file_path );
		$resized_url = $this->fix_domain_in_url( $upload_dir['baseurl'] . $resized_url );

		// Register resized image as new attachment.
		$new_attachment_id = $this->create_attachment_from_file( $resized_file_path, 0 );

		if ( ! $new_attachment_id ) {
			$this->socialwire_log( 'Failed to create new attachment, falling back to original' );
			return array(
				'url'           => $this->get_correct_attachment_url( $attachment_id ),
				'attachment_id' => $attachment_id,
			);
		}

		// Save resized image information in the original image's metadata (to prevent duplication).
		update_post_meta( $attachment_id, $resized_meta_key, $resized_url );
		update_post_meta( $attachment_id, $resized_meta_key . '_attachment_id', $new_attachment_id );

		// Save original image information in the new attachment (for association).
		update_post_meta( $new_attachment_id, 'pcn_original_attachment_id', $attachment_id );

		$this->socialwire_log( 'Successfully created resized/converted image: ' . $resized_url . ' (New attachment ID: ' . $new_attachment_id . ')' );
		return array(
			'url'           => $resized_url,
			'attachment_id' => $new_attachment_id,
		);
	}

	/**
	 * Check if the given file is an animated GIF.
	 *
	 * @param string $file_path File path to check.
	 */
	private function is_animated_gif( $file_path ) {
		if ( ! function_exists( 'imagecreatefromgif' ) ) {
			return false;
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Reading local file, not remote URL.
		$file_contents = file_get_contents( $file_path );
		if ( ! $file_contents ) {
			return false;
		}

		// Check for characteristic byte sequences of animated GIFs.
		$count = preg_match_all( '#\x00\x21\xF9\x04.{4}\x00(?:\x2C|\x21)#s', $file_contents );
		return $count > 1;
	}

	/**
	 * Determine output format.
	 *
	 * @param string $target_format      Target format.
	 * @param string $original_mime_type Original MIME type.
	 */
	private function determine_output_format( $target_format, $original_mime_type ) {
		if ( 'original' === $target_format ) {
			return $original_mime_type;
		}

		$format_map = array(
			'jpg'  => 'image/jpeg',
			'jpeg' => 'image/jpeg',
			'webp' => 'image/webp',
			'png'  => 'image/png',
			'gif'  => 'image/gif',
		);

		return isset( $format_map[ $target_format ] ) ? $format_map[ $target_format ] : $original_mime_type;
	}

	/**
	 * Get the MIME type for a given file extension.
	 *
	 * @param string $mime_type MIME type.
	 */
	private function get_extension_for_format( $mime_type ) {
		$extension_map = array(
			'image/jpeg' => 'jpg',
			'image/webp' => 'webp',
			'image/png'  => 'png',
			'image/gif'  => 'gif',
		);

		return isset( $extension_map[ $mime_type ] ) ? $extension_map[ $mime_type ] : 'jpg';
	}

	/**
	 * Check if background conversion is needed.
	 *
	 * @param string $output_format      Output format.
	 * @param string $original_mime_type Original MIME type.
	 */
	private function needs_background_conversion( $output_format, $original_mime_type ) {
		// JPEG does not support transparency.
		if ( 'image/jpeg' === $output_format ) {
			// If the original image supports transparency.
			return in_array( $original_mime_type, array( 'image/png', 'image/gif', 'image/webp' ), true );
		}
		return false;
	}

	/**
	 * Convert transparent background to white background.
	 *
	 * @param object $_image_editor WordPress image editor instance (currently unused, reserved for future enhancement).
	 */
	private function convert_transparent_to_white( $_image_editor ) {
		// Suppress unused parameter warning - parameter reserved for future functionality.
		unset( $_image_editor );

		// In WordPress image editor, transparent backgrounds are automatically converted to white when saved as JPEG.
		// If additional processing is needed, implement it here.
		$this->socialwire_log( 'Transparent background will be converted to white during JPEG save' );
	}

	/**
	 * Download images from post content and save them to the WordPress media library.
	 *
	 * @param int    $post_id Post ID.
	 * @param string $content Content with images.
	 */
	private function download_content_images( $post_id, $content ) {
		$this->socialwire_log( 'download_content_images called for post ID: ' . $post_id );

		if ( empty( $content ) ) {
			$this->socialwire_log( 'Content is empty, returning as-is' );
			return $content;
		}

		// Convert <h1> to <h2>.
		$content = $this->convert_h1_to_h2( $content );

		$pattern = '/<img[^>]+src=["\']([^"\']+)["\'][^>]*>/i';

		if ( preg_match_all( $pattern, $content, $matches, PREG_SET_ORDER ) ) {
			$this->socialwire_log( 'Found ' . count( $matches ) . ' images in content' );

			$updated_content  = $content;
			$failed_downloads = array();

			foreach ( $matches as $index => $match ) {
				$full_img_tag = $match[0];
				$image_url    = $match[1];

				$this->socialwire_log( 'Processing image ' . ( $index + 1 ) . ': ' . $image_url );

				if ( strpos( $image_url, wp_upload_dir()['baseurl'] ) !== false ) {
					$this->socialwire_log( 'Image already in uploads directory, skipping' );
					continue;
				}

				if ( strpos( $image_url, 'http' ) !== 0 ) {
					$this->socialwire_log( 'Image URL does not start with http, removing img tag' );
					$failed_downloads[] = $full_img_tag;
					continue;
				}

				$attachment_id = $this->download_image_to_media_library_safe( $image_url, $post_id );
				$this->socialwire_log( 'download_image_to_media_library_safe returned ID: ' . ( $attachment_id ? $attachment_id : 'false' ) );

				if ( $attachment_id ) {
					// Resize and convert format based on content image settings.
					$resize_result = $this->resize_image_if_needed( $attachment_id, 'content' );
					$this->socialwire_log( 'New image URL after resize: ' . ( $resize_result['url'] ? $resize_result['url'] : 'empty' ) );

					if ( $resize_result['url'] ) {
						$new_image_url = $resize_result['url'];
						$this->socialwire_log( 'Replacing: ' . $image_url . ' with: ' . $new_image_url );

						// If modal feature is enabled, change the image tag.
						$content_settings = get_option( 'socialwire_content_image_settings', array() );
						$enable_modal     = isset( $content_settings['enable_image_modal'] ) ? $content_settings['enable_image_modal'] : false;

						if ( $enable_modal ) {
							// Get original image URL (for modal display).
							$original_image_url = $this->get_correct_attachment_url( $attachment_id );
							$new_img_tag        = $this->create_modal_image_tag( $full_img_tag, $new_image_url, $original_image_url );
							$updated_content    = str_replace( $full_img_tag, $new_img_tag, $updated_content );
						} else {
							// Even if modal is disabled, add custom class if specified.
							$new_img_tag = $this->add_additional_image_class( $full_img_tag );
							$new_img_tag = preg_replace( '/src=["\'][^"\']*["\']/', 'src="' . esc_attr( $new_image_url ) . '"', $new_img_tag );
							$updated_content = str_replace( $full_img_tag, $new_img_tag, $updated_content );
						}
					} else {
						$this->socialwire_log( 'New image URL is empty, removing img tag' );
						$failed_downloads[] = $full_img_tag;
					}
				} else {
					$this->socialwire_log( 'Failed to download image, removing img tag: ' . $image_url );
					$failed_downloads[] = $full_img_tag;
				}
			}

			// Remove failed image tags.
			if ( ! empty( $failed_downloads ) ) {
				foreach ( $failed_downloads as $failed_tag ) {
					$updated_content = str_replace( $failed_tag, '', $updated_content );
				}
				$this->socialwire_log( 'Removed ' . count( $failed_downloads ) . ' failed image tags from content' );
			}

			$this->socialwire_log( 'Content processing completed' );
			return $updated_content;
		}

		$this->socialwire_log( 'No images found in content' );
		return $content;
	}

	/**
	 * Download image (with duplicate prevention).
	 *
	 * @param string $image_url Image URL to download.
	 * @param int    $post_id   Post ID.
	 */
	private function download_image_to_media_library_safe( $image_url, $post_id ) {
		if ( empty( $image_url ) ) {
			return false;
		}

		$this->socialwire_log( 'Checking for existing image: ' . $image_url );
		$existing_attachment = $this->get_attachment_by_url( $image_url );
		if ( $existing_attachment ) {
			$this->socialwire_log( 'Image already exists in media library: ' . $image_url . ' (ID: ' . $existing_attachment->ID . ')' );
			return $existing_attachment->ID;
		}

		// Acquire lock for image URL.
		$image_hash = md5( $image_url );
		$lock_id    = $this->acquire_distributed_lock( 'image_' . $image_hash, 300 );

		if ( ! $lock_id ) {
			$this->socialwire_log( 'Another process is downloading the same image, waiting...' );
			sleep( 2 );

			// Check again.
			$existing_attachment = $this->get_attachment_by_url( $image_url );
			if ( $existing_attachment ) {
				$this->socialwire_log( 'Image was downloaded by another process: ' . $image_url . ' (ID: ' . $existing_attachment->ID . ')' );
				return $existing_attachment->ID;
			}

			$this->socialwire_log( 'Could not acquire image lock, skipping: ' . $image_url );
			return false;
		}

		try {
			// Double-check for existing image after lock acquisition.
			$this->socialwire_log( 'Double-checking for existing image after lock acquisition: ' . $image_url );
			$existing_attachment = $this->get_attachment_by_url( $image_url );
			if ( $existing_attachment ) {
				$this->socialwire_log( 'Image already exists (double-check): ' . $image_url . ' (ID: ' . $existing_attachment->ID . ')' );
				return $existing_attachment->ID;
			}

			$this->socialwire_log( 'Proceeding with image download: ' . $image_url );

			require_once ABSPATH . 'wp-admin/includes/media.php';
			require_once ABSPATH . 'wp-admin/includes/file.php';
			require_once ABSPATH . 'wp-admin/includes/image.php';

			add_filter(
				'http_request_timeout',
				function () {
					return 60;
				}
			);

			$attachment_id = media_sideload_image( $image_url, $post_id, null, 'id' );

			if ( is_wp_error( $attachment_id ) ) {
				$this->socialwire_log( 'Image download error: ' . $image_url . ' - ' . $attachment_id->get_error_message() );
				return false;
			}

			// Save original URL to meta field (to prevent duplicates).
			update_post_meta( $attachment_id, 'pcn_original_image_url', $image_url );
			$this->socialwire_log( 'Saved original URL to meta field for attachment ID: ' . $attachment_id );

			$this->socialwire_log( 'Successfully downloaded image: ' . $image_url . ' (ID: ' . $attachment_id . ')' );
			return $attachment_id;
		} finally {
			if ( $lock_id ) {
				$this->release_distributed_lock( 'image_' . $image_hash, $lock_id );
			}
		}
	}

	/**
	 * Get existing attachment by image URL.
	 *
	 * @param string $image_url 画像URL.
	 */
	private function get_attachment_by_url( $image_url ) {
		global $wpdb;

		// First, search by original URL in meta field.
		// Direct database query is necessary for performance with large attachment libraries.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
		$attachment_ids = $wpdb->get_col(
			$wpdb->prepare(
				"
            SELECT post_id FROM {$wpdb->postmeta} 
            WHERE meta_key = 'pcn_original_image_url' 
            AND meta_value = %s
        ",
				$image_url
			)
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( ! empty( $attachment_ids ) ) {
			$this->socialwire_log( 'Found existing attachment by meta field for URL: ' . $image_url . ' (ID: ' . $attachment_ids[0] . ')' );
			return get_post( $attachment_ids[0] );
		}

		// Legacy guid search for backward compatibility.
		// Direct database query is necessary for performance with large attachment libraries.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
		$attachment = $wpdb->get_col(
			$wpdb->prepare(
				"
            SELECT ID FROM {$wpdb->posts} 
            WHERE post_type = 'attachment' 
            AND guid = %s
        ",
				$image_url
			)
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( ! empty( $attachment ) ) {
			$this->socialwire_log( 'Found existing attachment by guid for URL: ' . $image_url . ' (ID: ' . $attachment[0] . ')' );
			return get_post( $attachment[0] );
		}

		return null;
	}

	/**
	 * Get correct attachment URL.
	 *
	 * @param int $attachment_id Attachment ID.
	 */
	private function get_correct_attachment_url( $attachment_id ) {
		if ( ! $attachment_id ) {
			return false;
		}

		$this->socialwire_log( 'get_correct_attachment_url called with ID: ' . $attachment_id );

		$upload_dir = wp_upload_dir();
		$file_path  = get_attached_file( $attachment_id );

		$this->socialwire_log( 'Upload dir baseurl: ' . $upload_dir['baseurl'] );
		$this->socialwire_log( 'File path: ' . ( $file_path ? $file_path : 'null' ) );

		if ( $file_path ) {
			$relative_path = str_replace( $upload_dir['basedir'], '', $file_path );
			$url           = $upload_dir['baseurl'] . $relative_path;

			$this->socialwire_log( 'Generated URL from file path: ' . $url );
			$fixed_url = $this->fix_domain_in_url( $upload_dir['baseurl'] . $relative_path );
			$this->socialwire_log( 'After fix_domain_in_url: ' . $fixed_url );
			return $fixed_url;
		}

		$url = wp_get_attachment_url( $attachment_id );
		$this->socialwire_log( 'wp_get_attachment_url returned: ' . ( $url ? $url : 'false' ) );
		if ( ! $url ) {
			$this->socialwire_log( 'Could not get attachment URL for ID: ' . $attachment_id );
			return false;
		}

		$fixed_url = $this->fix_domain_in_url( $url );
		$this->socialwire_log( 'After fix_domain_in_url: ' . $fixed_url );
		return $fixed_url;
	}

	/**
	 * Fix domain in URL.
	 *
	 * @param string $url URL to be corrected.
	 */
	private function fix_domain_in_url( $url ) {
		$this->socialwire_log( 'fix_domain_in_url called with: ' . ( $url ? $url : 'empty' ) );

		if ( empty( $url ) ) {
			$this->socialwire_log( 'URL is empty, returning as-is' );
			return $url;
		}

		$original_url = $url;
		$parsed_url   = wp_parse_url( $url );

		if ( ! $parsed_url ) {
			$this->socialwire_log( 'Failed to parse URL' );
			return $url;
		}

		$path = isset( $parsed_url['path'] ) ? $parsed_url['path'] : '';
		$this->socialwire_log( 'Extracted path: ' . $path );

		if ( strpos( $path, '/wp-content/uploads/' ) === 0 ) {
			$this->socialwire_log( 'Converting to relative path: ' . $original_url . ' -> ' . $path );
			return $path;
		}

		$this->socialwire_log( 'Path does not start with /wp-content/uploads/, no conversion needed' );
		return $url;
	}

	/**
	 * Log output function.
	 *
	 * @param string $message Log message.
	 */
	private function socialwire_log( $message ) {

		// Save debug logs to array when executed manually.
		if ( isset( $this->debug_logs ) ) {
			$this->debug_logs[] = '[' . current_time( 'Y-m-d H:i:s' ) . '] ' . $message;
		}
	}

	/**
	 * Manual edit detection (on post update).
	 *
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post_after Updated post object.
	 * @param WP_Post $_post_before Post object before update (unused in current implementation).
	 */
	public function detect_manual_edit( $post_id, $post_after, $_post_before ) {
		// Suppress unused parameter warning.
		unset( $_post_before );

		// Check if the post is a PCN article.
		if ( ! get_post_meta( $post_id, 'pcn_document_id', true ) ) {
			return;
		}

		// Check if the update is during RSS import.
		if ( $this->is_rss_import_in_progress() ) {
			return;
		}

		// Check if the update is from the admin panel or REST API.
		if ( is_admin() || defined( 'REST_REQUEST' ) ) {
			$this->socialwire_log( 'Manual edit detected for post ID: ' . $post_id );

			// Set manual edit flag.
			update_post_meta( $post_id, 'pcn_manually_edited', current_time( 'mysql' ) );
			update_post_meta( $post_id, 'pcn_manual_edit_type', 'content_update' );

			// Save original content hash (for comparison).
			$content_hash = md5( $post_after->post_title . $post_after->post_content . $post_after->post_excerpt );
			update_post_meta( $post_id, 'pcn_manual_content_hash', $content_hash );

			$this->socialwire_log( 'Post ID ' . $post_id . ' marked as manually edited' );
		}
	}

	/**
	 * Manual delete detection (supports both trashing and permanent deletion).
	 *
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post Post object (optional).
	 */
	public function detect_manual_delete( $post_id, $post = null ) {
		// If $post is null (in case of wp_trash_post), get the post.
		if ( null === $post ) {
			$post = get_post( $post_id );
		}

		if ( ! $post ) {
			$this->socialwire_log( 'Could not get post data for ID: ' . $post_id );
			return;
		}

		// Check if the post is a PCN article.
		$document_id = get_post_meta( $post_id, 'pcn_document_id', true );
		if ( ! $document_id ) {
			return;
		}

		// Check if the deletion is during RSS import.
		if ( $this->is_rss_import_in_progress() ) {
			$this->socialwire_log( 'Skipping delete detection during RSS import for post ID: ' . $post_id );
			return;
		}

		$this->socialwire_log( 'Manual delete detected for PCN post ID: ' . $post_id . ' (Document ID: ' . $document_id . ')' );

		// Record the document_id of the deleted post (to skip in future RSS fetches).
		$deleted_ids                 = get_option( 'pcn_manually_deleted_document_ids', array() );
		$deleted_ids[ $document_id ] = array(
			'deleted_at' => current_time( 'mysql' ),
			'post_title' => $post->post_title,
			'action'     => current_action(), // wp_trash_post or before_delete_post.
		);
		update_option( 'pcn_manually_deleted_document_ids', $deleted_ids );

		$action_type = current_action() === 'wp_trash_post' ? 'moved to trash' : 'permanently deleted';
		$this->socialwire_log( 'Added document_id to manually deleted list: ' . $document_id . ' (' . $action_type . ')' );
	}

	/**
	 * Check if RSS import is in progress.
	 */
	private function is_rss_import_in_progress() {
		// Check transient indicating RSS import is in progress.
		return get_transient( 'socialwire_rss_import_in_progress' ) === true;
	}

	/**
	 * Check if the post is manually edited.
	 *
	 * @param int $post_id Post ID.
	 */
	private function is_manually_edited( $post_id ) {
		$manually_edited = get_post_meta( $post_id, 'pcn_manually_edited', true );
		return ! empty( $manually_edited );
	}

	/**
	 * Check if the document ID is manually deleted.
	 *
	 * @param string $document_id Document ID.
	 */
	private function is_manually_deleted_document_id( $document_id ) {
		$deleted_ids = get_option( 'pcn_manually_deleted_document_ids', array() );
		return isset( $deleted_ids[ $document_id ] );
	}

	/**
	 * Get the number of articles created today (WordPress timezone).
	 *
	 * @return int Number of articles created today.
	 */
	private function get_articles_created_today() {
		// Get today's date in WordPress timezone (fixed: separate date and time)
		$today_date = current_time( 'Y-m-d' );
		$today_start = strtotime( $today_date . ' 00:00:00' );
		$today_end = strtotime( $today_date . ' 23:59:59' );

		// Convert to GMT for WP_Query
		$today_start_gmt = gmdate( 'Y-m-d H:i:s', $today_start );
		$today_end_gmt = gmdate( 'Y-m-d H:i:s', $today_end );

		// Debug logging for date calculation
		$this->socialwire_log( sprintf(
			'get_articles_created_today: WP timezone date=%s, Query range GMT: %s to %s',
			$today_date,
			$today_start_gmt,
			$today_end_gmt
		) );

		$args = array(
			'post_type'      => 'post',
			'post_status'    => array( 'publish', 'draft', 'pending', 'private', 'trash' ),
			'meta_query'     => array(
				array(
					'key'     => 'pcn_generated',
					'value'   => '1',
					'compare' => '=',
				),
			),
			'date_query'     => array(
				array(
					'after'     => $today_start_gmt,
					'before'    => $today_end_gmt,
					'inclusive' => true,
					'column'    => 'post_date_gmt',
				),
			),
			'fields'         => 'ids',
			'posts_per_page' => -1,
		);

		$query = new WP_Query( $args );

		$this->socialwire_log( sprintf( 'get_articles_created_today: Found %d articles', $query->found_posts ) );

		return $query->found_posts;
	}

	/**
	 * Check if daily article limit has been reached.
	 *
	 * @return bool True if limit reached, false otherwise.
	 */
	private function is_daily_article_limit_reached() {
		$settings = get_option( $this->option_name, array() );
		$max_articles_per_day = isset( $settings['max_articles_per_day'] ) ? intval( $settings['max_articles_per_day'] ) : 0;
		
		// 0 means no limit
		if ( $max_articles_per_day === 0 ) {
			return false;
		}
		
		$articles_created_today = $this->get_articles_created_today();
		
		$this->socialwire_log( sprintf( 'Daily article limit check: %d/%d articles created today', $articles_created_today, $max_articles_per_day ) );
		
		return $articles_created_today >= $max_articles_per_day;
	}


	/**
	 * Adjust heading levels.
	 *
	 * @param string $content Content.
	 * @param string $start_level Start level.
	 */
	private function adjust_heading_levels( $content, $start_level = 'h2' ) {
		if ( empty( $content ) ) {
			return $content;
		}

		$this->socialwire_log( 'Adjusting heading levels to start from: ' . $start_level );

		// Convert start level to a number.
		$start_num = intval( substr( $start_level, 1 ) ); // h2 → 2, h3 → 3, etc.

		// Detect and convert existing headings from h2 to h6.
		$heading_pattern = '/<(h[2-6])(\s[^>]*)?>(.*?)<\/h[2-6]>/i';

		$adjusted_content = preg_replace_callback(
			$heading_pattern,
			function ( $matches ) use ( $start_num ) {
				$original_tag = $matches[1]; // h2, h3, etc.
				$attributes   = $matches[2] ?? ''; // Attributes if any.
				$content      = $matches[3]; // Heading content.

				// Get the original level.
				$original_level = intval( substr( $original_tag, 1 ) );

				// Calculate the new level (adjusted based on h2).
				$level_diff = $original_level - 2; // Calculate the difference from the base h2.
				$new_level  = $start_num + $level_diff;

				// Limit to h6.
				if ( $new_level > 6 ) {
					$new_level = 6;
				}

				$new_tag = 'h' . $new_level;

				$this->socialwire_log( 'Converting ' . $original_tag . ' to ' . $new_tag );

				return '<' . $new_tag . $attributes . '>' . $content . '</' . $new_tag . '>';
			},
			$content
		);

		$this->socialwire_log( 'Heading level adjustment completed' );

		return $adjusted_content;
	}

	/**
	 * Link conversion process (with open in new tab setting support).
	 *
	 * @param string $content Content.
	 */
	private function convert_links_with_new_tab_support( $content ) {
		// Get link settings.
		$link_settings   = get_option( 'pcn_link_settings', array() );
		$open_in_new_tab = isset( $link_settings['open_in_new_tab'] ) ? $link_settings['open_in_new_tab'] : false;

		// Convert Markdown link format [text](url) to <a> tag.
		if ( $open_in_new_tab ) {
			// If the setting to open in a new tab is enabled.
			$content = preg_replace( '/\[([^\]]+)\]\(([^)]+)\)/', '<a href="$2" target="_blank" rel="noopener">$1</a>', $content );
		} else {
			// Normal link conversion.
			$content = preg_replace( '/\[([^\]]+)\]\(([^)]+)\)/', '<a href="$2">$1</a>', $content );
		}

		// Apply new tab setting to existing HTML link tags.
		if ( $open_in_new_tab ) {
			$content = $this->add_target_blank_to_existing_links( $content );
		}

		return $content;
	}

	/**
	 * Post content filter - Apply link settings on display.
	 *
	 * @param string $content Content.
	 */
	public function filter_content_links( $content ) {
		// Admin area - do not apply.
		if ( is_admin() ) {
			return $content;
		}

		// Check if the post has a document ID.
		global $post;
		if ( ! $post || ! get_post_meta( $post->ID, 'pcn_document_id', true ) ) {
			return $content;
		}

		return $this->apply_new_tab_to_links( $content );
	}

	/**
	 * Apply new tab setting to all links in content.
	 *
	 * @param string $content Content.
	 */
	private function apply_new_tab_to_links( $content ) {
		// Get link settings.
		$link_settings   = get_option( 'pcn_link_settings', array() );
		$open_in_new_tab = isset( $link_settings['open_in_new_tab'] ) ? $link_settings['open_in_new_tab'] : false;

		if ( ! $open_in_new_tab ) {
			return $content; // If the setting is disabled, return as is.
		}

		$this->socialwire_log( 'Applying new tab setting to all links in content' );

		return $this->add_target_blank_to_existing_links( $content );
	}

	/**
	 * Add target="_blank" to existing link tags.
	 *
	 * @param string $content Content.
	 */
	private function add_target_blank_to_existing_links( $content ) {
		// Counter for logging the number of changes made.
		$changed_count = 0;

		// Use preg_replace_callback to process each <a> tag.
		// This regex matches <a> tags and captures their attributes.
		$content = preg_replace_callback(
			'/<a\s+([^>]*)>/i',
			function ( $matches ) use ( &$changed_count ) {
				$full_match = $matches[0];
				$attributes = $matches[1];

				// If the target attribute is already present, return as is.
				if ( preg_match( '/target\s*=\s*["\'][^"\']*["\']/i', $full_match ) ) {
					return $full_match;
				}

				// If the href attribute is not present, return as is.
				if ( ! preg_match( '/href\s*=\s*["\'][^"\']*["\']/i', $attributes ) ) {
					return $full_match;
				}

				$changed_count++;
				// Add target="_blank" rel="noopener".
				// Consider existing rel attribute if present.
				if ( preg_match( '/rel\s*=\s*["\']([^"\']*)["\']/i', $attributes, $rel_matches ) ) {
					// If noopener is not already present, add it.
					$existing_rel = $rel_matches[1];
					if ( strpos( $existing_rel, 'noopener' ) === false ) {
						$new_rel    = trim( $existing_rel . ' noopener' );
						$attributes = preg_replace( '/rel\s*=\s*["\']([^"\']*)["\']/i', 'rel="' . $new_rel . '"', $attributes );
					}
					return '<a ' . $attributes . ' target="_blank">';
				} else {
					// If rel attribute is not present, add it.
					return '<a ' . $attributes . ' target="_blank" rel="noopener">';
				}
			},
			$content
		);

		if ( $changed_count > 0 ) {
			$this->socialwire_log( 'Added target="_blank" to ' . $changed_count . ' links' );
		}

		return $content;
	}

	/**
	 * Create a modal image tag.
	 *
	 * @param string $original_img_tag The original image tag.
	 * @param string $new_image_url The new image URL.
	 * @param string $original_image_url The original image URL.
	 */
	private function create_modal_image_tag( $original_img_tag, $new_image_url, $original_image_url ) {
		// Get additional image class from settings.
		$content_settings = get_option( 'socialwire_content_image_settings', array() );
		$additional_class = isset( $content_settings['additional_image_class'] ) ? trim( $content_settings['additional_image_class'] ) : '';
		
		// Build class attribute.
		$classes = 'pcn-modal-image';
		if ( ! empty( $additional_class ) ) {
			$classes .= ' ' . $additional_class;
		}

		// Create a modal image tag.
		$modal_img_tag = str_replace(
			'<img',
			'<img class="' . esc_attr( $classes ) . '" data-full-src="' . esc_attr( $original_image_url ) . '"',
			$original_img_tag
		);

		// Change src to the new URL.
		$modal_img_tag = preg_replace( '/src=["\'][^"\']*["\']/', 'src="' . esc_attr( $new_image_url ) . '"', $modal_img_tag );

		return $modal_img_tag;
	}

	/**
	 * Add additional class to image tag (for non-modal images).
	 *
	 * @param string $img_tag The original image tag.
	 * @return string Modified image tag.
	 */
	private function add_additional_image_class( $img_tag ) {
		// Get additional image class from settings.
		$content_settings = get_option( 'socialwire_content_image_settings', array() );
		$additional_class = isset( $content_settings['additional_image_class'] ) ? trim( $content_settings['additional_image_class'] ) : '';
		
		// If no additional class specified, return original tag.
		if ( empty( $additional_class ) ) {
			return $img_tag;
		}

		// Check if class attribute already exists.
		if ( preg_match( '/class\s*=\s*["\']([^"\']*)["\']/', $img_tag, $matches ) ) {
			// Add to existing class.
			$existing_classes = $matches[1];
			$new_classes = trim( $existing_classes . ' ' . $additional_class );
			return preg_replace( '/class\s*=\s*["\']([^"\']*)["\']/', 'class="' . esc_attr( $new_classes ) . '"', $img_tag );
		} else {
			// Add new class attribute.
			return str_replace( '<img', '<img class="' . esc_attr( $additional_class ) . '"', $img_tag );
		}
	}

	/**
	 * Load scripts/styles for the frontend.
	 */
	public function frontend_enqueue_scripts() {
		// Check if modal functionality is enabled.
		$content_settings = get_option( 'socialwire_content_image_settings', array() );
		$enable_modal     = isset( $content_settings['enable_image_modal'] ) ? $content_settings['enable_image_modal'] : false;

		if ( ! $enable_modal ) {
			return;
		}

		// Ensure jQuery is loaded.
		wp_enqueue_script( 'jquery' );

		// Register and enqueue CSS for modal.
		$css_handle = 'socialwire-modal-css';
		wp_register_style( $css_handle, false );
		wp_enqueue_style( $css_handle );
		wp_add_inline_style( $css_handle, $this->get_modal_css() );

		// Add inline JavaScript.
		wp_add_inline_script( 'jquery', $this->get_modal_javascript() );
	}



	/**
	 * Get modal CSS styles.
	 */
	private function get_modal_css() {
		return '
        .pcn-modal-image {
            cursor: pointer;
            transition: opacity 0.2s;
        }
        .pcn-modal-image:hover {
            opacity: 0.8;
        }
        .pcn-image-modal {
            display: none;
            position: fixed;
            z-index: 999999;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.8);
            backdrop-filter: blur(3px);
        }
        .pcn-modal-content {
            position: relative;
            margin: auto;
            display: block;
            max-width: 90%;
            max-height: 90%;
            top: 50%;
            transform: translateY(-50%);
        }
        .pcn-modal-image-full {
            width: auto;
            height: auto;
            max-width: 100%;
            max-height: 100%;
            display: block;
            margin: auto;
        }
        .pcn-modal-close {
            position: absolute;
            top: 20px;
            right: 35px;
            color: #fff;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
            z-index: 1000000;
        }
        .pcn-modal-close:hover {
            opacity: 0.7;
        }
        ';
	}



	/**
	 * Get modal JavaScript code.
	 */
	private function get_modal_javascript() {
		return '
        jQuery(document).ready(function($) {
            $(document).on("click", ".pcn-modal-image", function(e) {
                e.preventDefault();
                var fullSrc = $(this).data("full-src");
                if (fullSrc) {
                    $("#pcn-image-modal-img").attr("src", fullSrc);
                    $("#pcn-image-modal").show();
                }
            });
            
            $(document).on("click", ".pcn-modal-close, .pcn-image-modal", function(e) {
                if (e.target === this) {
                    $("#pcn-image-modal").hide();
                }
            });
            
            $(document).keydown(function(e) {
                if (e.keyCode === 27 && $("#pcn-image-modal").is(":visible")) {
                    $("#pcn-image-modal").hide();
                }
            });
        });
        ';
	}

	/**
	 * Add modal HTML to the footer.
	 */
	public function add_modal_html() {
		// Check if modal functionality is enabled.
		$content_settings = get_option( 'socialwire_content_image_settings', array() );
		$enable_modal     = isset( $content_settings['enable_image_modal'] ) ? $content_settings['enable_image_modal'] : false;

		if ( ! $enable_modal ) {
			return;
		}

		echo '
        <div id="pcn-image-modal" class="pcn-image-modal">
            <span class="pcn-modal-close">&times;</span>
            <div class="pcn-modal-content">
                <img id="pcn-image-modal-img" class="pcn-modal-image-full" src="" alt="">
            </div>
        </div>
        ';
	}

	/**
	 * Create a new attachment from a file path.
	 *
	 * @param string $file_path The file path.
	 * @param int    $parent_post_id The parent post ID.
	 */
	private function create_attachment_from_file( $file_path, $parent_post_id = 0 ) {
		if ( ! file_exists( $file_path ) ) {
			$this->socialwire_log( 'File does not exist: ' . $file_path );
			return false;
		}

		$file_type  = wp_check_filetype( basename( $file_path ), null );
		$upload_dir = wp_upload_dir();

		$attachment = array(
			'guid'           => $upload_dir['baseurl'] . '/' . basename( $file_path ),
			'post_mime_type' => $file_type['type'],
			'post_title'     => preg_replace( '/\.[^.]+$/', '', basename( $file_path ) ),
			'post_content'   => '',
			'post_status'    => 'inherit',
		);

		$attachment_id = wp_insert_attachment( $attachment, $file_path, $parent_post_id );

		if ( is_wp_error( $attachment_id ) ) {
			$this->socialwire_log( 'Failed to create attachment: ' . $attachment_id->get_error_message() );
			return false;
		}

		// Generate and save metadata.
		require_once ABSPATH . 'wp-admin/includes/image.php';
		$attachment_data = wp_generate_attachment_metadata( $attachment_id, $file_path );
		wp_update_attachment_metadata( $attachment_id, $attachment_data );

		$this->socialwire_log( 'Created new attachment ID: ' . $attachment_id . ' for file: ' . $file_path );
		return $attachment_id;
	}
}

/**
 * Modify all plugins array for localization.
 *
 * @param array $all_plugins All plugins array.
 * @return array Modified plugins array.
 */
// phpcs:disable Universal.Files.SeparateFunctionsFromOO.Mixed

/**
 * Modify all plugins description for localization.
 *
 * @param array $all_plugins Array of all plugins.
 * @return array Modified plugins array.
 */
function socialwire_modify_all_plugins_description( $all_plugins ) {
	// Ensure translation domain is loaded before translating plugin metadata.
	socialwire_load_plugin_textdomain();
	
	foreach ( $all_plugins as $plugin_file => $plugin_data ) {
		if ( strpos( $plugin_file, 'socialwire-article-generator' ) !== false ) {
			// Get user locale and force Japanese translation if user locale is Japanese
			$user_locale = function_exists( 'get_user_locale' ) ? get_user_locale() : 'en_US';
			
			if ( $user_locale === 'ja' || strpos( $user_locale, 'ja' ) === 0 ) {
				// Force Japanese translation - exact match to PO file
				$all_plugins[ $plugin_file ]['Name']        = 'Socialwire Article Generator';
				$all_plugins[ $plugin_file ]['Description'] = 'プレスリリースから自動的に記事を生成し、効率的なメディア管理をサポートするAI搭載のWordPressプラグイン。';
			} else {
				// Use standard translation function with exact PO file string
				$all_plugins[ $plugin_file ]['Name']        = __( 'Socialwire Article Generator', 'socialwire-article-generator' );
				$all_plugins[ $plugin_file ]['Description'] = __( 'AI-powered WordPress plugin that automatically generates articles from press releases to support efficient media management.', 'socialwire-article-generator' );
			}
			$all_plugins[ $plugin_file ]['Author']      = 'SocialWire';
			break;
		}
	}

	return $all_plugins;
}

/**
 * Modify plugin description based on site language.
 *
 * @param array  $plugin_data Plugin data array.
 * @param string $plugin_file Plugin file path.
 * @return array Modified plugin data.
 */
function socialwire_modify_plugin_description( $plugin_data, $plugin_file ) {
	// Check if this is our plugin.
	if ( strpos( $plugin_file, 'socialwire-article-generator' ) === false ) {
		return $plugin_data;
	}

	// Ensure translation domain is loaded before translating plugin metadata.
	socialwire_load_plugin_textdomain();

	// Get user locale and force Japanese translation if user locale is Japanese
	$user_locale = function_exists( 'get_user_locale' ) ? get_user_locale() : 'en_US';
	
	if ( $user_locale === 'ja' || strpos( $user_locale, 'ja' ) === 0 ) {
		// Force Japanese translation - exact match to PO file
		$plugin_data['Name']        = 'Socialwire Article Generator';
		$plugin_data['Description'] = 'プレスリリースから自動的に記事を生成し、効率的なメディア管理をサポートするAI搭載のWordPressプラグイン。';
	} else {
		// Use standard translation function with exact PO file string
		$plugin_data['Name']        = __( 'Socialwire Article Generator', 'socialwire-article-generator' );
		$plugin_data['Description'] = __( 'AI-powered WordPress plugin that automatically generates articles from press releases to support efficient media management.', 'socialwire-article-generator' );
	}
	$plugin_data['Author']      = 'SocialWire';

	return $plugin_data;
}

// Add filters to modify plugin information with high priority.
add_filter( 'all_plugins', 'socialwire_modify_all_plugins_description', 99 );

// Hook plugin meta translation very early
add_action( 'plugins_loaded', function() {
	// Force load text domain as early as possible
	socialwire_load_plugin_textdomain();
	
	// Add plugin row meta filter with high priority
	add_filter( 'plugin_row_meta', 'socialwire_modify_plugin_description', 99, 2 );
}, 1 );

// Also hook on admin_init for admin pages
add_action( 'admin_init', function() {
	// Ensure text domain is loaded on admin pages
	socialwire_load_plugin_textdomain();
}, 1 );

// Initialize the plugin. Memory optimized version.
// WordPress 4.6+ automatically loads translations for plugins hosted on WordPress.org.
if ( defined( 'ABSPATH' ) ) {
	// Early translation loading for proper internationalization.
	add_action( 'plugins_loaded', 'socialwire_load_plugin_textdomain', 1 );
	
	// WordPress standard plugin header translation hook
	add_filter( 'gettext', function( $translation, $text, $domain ) {
		if ( $domain === 'socialwire-article-generator' || $domain === 'default' ) {
			// Force load text domain
			static $loaded = false;
			if ( ! $loaded ) {
				socialwire_load_plugin_textdomain();
				$loaded = true;
			}
			
			// Handle plugin description translation
			if ( $text === 'AI-powered WordPress plugin that automatically generates articles from press releases to support efficient media management.' ) {
				$user_locale = function_exists( 'get_user_locale' ) ? get_user_locale() : get_locale();
				if ( $user_locale === 'ja' || strpos( $user_locale, 'ja' ) === 0 ) {
					return 'プレスリリースから自動的に記事を生成し、効率的なメディア管理をサポートするAI搭載のWordPressプラグイン。';
				}
			}
		}
		return $translation;
	}, 10, 3 );
	
	// Memory management only when actually needed.
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is for admin page detection only
	if ( is_admin() && isset( $_GET['page'] ) && $_GET['page'] === 'socialwire-generator-settings' ) {
		if ( function_exists( 'wp_raise_memory_limit' ) ) {
			wp_raise_memory_limit( 'admin' );
		}
	}
	
	new Socialwire_Article_Generator();
}

/**
 * Load plugin text domain early for proper translation support.
 */
function socialwire_load_plugin_textdomain() {
	$domain = 'socialwire-article-generator';
	
	// Force unload existing domain.
	if ( function_exists( 'is_textdomain_loaded' ) && is_textdomain_loaded( $domain ) ) {
		unload_textdomain( $domain );
	}
	
	// Get user locale if in admin.
	$locale = function_exists( 'determine_locale' ) ? determine_locale() : get_locale();
	if ( is_admin() && function_exists( 'get_user_locale' ) ) {
		$locale = get_user_locale();
	}
	
	// Define plugin directory.
	$plugin_dir = dirname( __FILE__ );
	$languages_dir = $plugin_dir . '/languages';
	
	// WordPress 4.6+ automatically loads translations, so we only do manual loading as fallback
	$loaded = false;
	
	// Method 1: Direct loading with user locale
	$mo_file = $languages_dir . '/' . $locale . '.mo';
	if ( file_exists( $mo_file ) ) {
		$loaded = load_textdomain( $domain, $mo_file );
	}
	
	// Method 2: Force Japanese if locale is Japanese and not yet loaded
	if ( ! $loaded && ( $locale === 'ja' || strpos( $locale, 'ja' ) === 0 ) ) {
		$ja_mo_file = $languages_dir . '/ja.mo';
		if ( file_exists( $ja_mo_file ) ) {
			$loaded = load_textdomain( $domain, $ja_mo_file );
		}
	}
	
	return $loaded;
}

/**
 * WordPress standard way to translate plugin headers.
 * This function is called automatically by WordPress when displaying plugin info.
 */
function socialwire_translate_plugin_headers() {
	// Ensure text domain is loaded
	socialwire_load_plugin_textdomain();
	
	// Return translated plugin information
	return array(
		'Name' => __( 'Socialwire Article Generator', 'socialwire-article-generator' ),
		'Description' => __( 'AI-powered WordPress plugin that automatically generates articles from press releases to support efficient media management.', 'socialwire-article-generator' ),
		'Author' => 'SocialWire'
	);
}
?>